import { HttpService } from '@midwayjs/axios';
import { Config, ILogger, Inject, Provide } from '@midwayjs/core';

@Provide()
export class GitlabService {
  @Inject()
  httpService: HttpService;

  @Config('gitlab')
  gitlabConfig;

  @Inject()
  logger: ILogger;

  // 获取用户所有组
  async getGroups() {
    const { host, apiToken } = this.gitlabConfig;
    const url = `${host}/api/v4/groups?private_token=${apiToken}&per_page=100`;
    const { data } = await this.httpService.get(url);
    return data;
  }

  // 获取组下所有项目
  async getProjects(groupId) {
    const { host, apiToken } = this.gitlabConfig;
    const url = `${host}/api/v4/groups/${groupId}/projects?private_token=${apiToken}&per_page=100`;
    const { data } = await this.httpService.get(url);
    return data;
  }

  // 获取项目下所有分支
  async getBranches(projectId) {
    const { host, apiToken } = this.gitlabConfig;
    const url = `${host}/api/v4/projects/${projectId}/repository/branches?private_token=${apiToken}&per_page=100`;
    const { data } = await this.httpService.get(url);
    return data;
  }

  // 获取分支下所有提交时间是今天的提交
  async getCommits(projectId, branchName) {
    const todayStart = new Date();
    todayStart.setHours(0, 0, 0, 0);
    const todayEnd = new Date();
    todayEnd.setHours(23, 59, 59, 999);

    // convert date to ISO string
    const todayStartStr = todayStart.toISOString();
    const todayEndStr = todayEnd.toISOString();
    const { host, apiToken } = this.gitlabConfig;
    const url = `${host}/api/v4/projects/${projectId}/repository/commits?private_token=${apiToken}&ref_name=${branchName}&since=${todayStartStr}&until=${todayEndStr}&per_page=100`;
    const { data } = await this.httpService.get(url);
    return data;
  }

  // 获取提交详情
  async getCommitDetail(projectId, commitId) {
    const { host, apiToken } = this.gitlabConfig;
    const url = `${host}/api/v4/projects/${projectId}/repository/commits/${commitId}?private_token=${apiToken}`;
    const { data } = await this.httpService.get(url);
    return data;
  }

  async getCodeStat() {
    const groups = await this.getGroups();
    // console.log(groups);
    // 筛选出fe_arch、APP、ui组
    const uiGroup = groups.filter(
      group =>
        group.name === 'fe_arch' || group.name === 'APP' || group.name === 'ui'
    );
    console.log(uiGroup);
    const result = [];
    for (const group of uiGroup) {
      const projects = await this.getProjects(group.id);
      for (const project of projects) {
        let branches = await this.getBranches(project.id);
        // 过滤最后一次提交时间是在今天的分支
        branches = branches.filter(branch => {
          const lastCommitDate = new Date(branch.commit.committed_date);
          return lastCommitDate.toDateString() === new Date().toDateString();
        });

        console.log(
          `组${group.name} 项目${project.name} 今日有提交的分支：`,
          branches.map(item => item.name)
        );
        for (const branch of branches) {
          let commits = await this.getCommits(project.id, branch.name);
          // 忽略 merge commits
          commits = commits.filter(
            commit =>
              !commit.title.startsWith('Merge') &&
              !commit.message.startsWith('Merge')
          );

          for (const commit of commits) {
            const commitDetail = await this.getCommitDetail(
              project.id,
              commit.id
            );
            result.push(commitDetail);
          }
        }
      }
    }
    // 按作者分组
    const authorMap = {};
    for (const commit of result) {
      const author = commit.author_name;
      if (!authorMap[author]) {
        authorMap[author] = {
          name: author,
          commits: [],
          additions: 0,
          deletions: 0,
        };
      }
      authorMap[author].commits.push(commit);
      authorMap[author].additions += commit.stats.additions;
      authorMap[author].deletions += commit.stats.deletions;
    }
    // 按增加行数排序
    const authorList = Object.values(authorMap).sort(
      (a, b) => b['additions'] - a['additions']
    );
    return authorList;
  }
}
