import axios from 'axios';
import strategyModes from './config';

const ERR_MESSAGE_MAP = {
  status: {
    400: '错误请求',
    401: '您未登录或登录超时，请重新登录',
    403: '拒绝访问',
    404: '请求错误,未找到该资源',
    405: '请求方法未允许',
    408: '请求超时',
    500: '服务器端出错',
    501: '网络未实现',
    502: '网络错误',
    503: '服务不可用',
    504: '网络超时',
    505: 'http版本不支持该请求'
  }
};
const defaultStratege = {
  default: {
    request(cfg) {
      return cfg;
    },
    response(res) {
      return res.data;
    }
  }
};

const defaultConfig = {
  timeout: 15000,
  strategy: 'service'
};
class HttpRequest {
  constructor(strategy = {}, config = {}, toastFn = () => {}, loadingFn = () => {}) {
    this.CancelToken = axios.CancelToken;
    this.axiosConfig = { ...defaultConfig, ...config };
    this.instance = axios.create(this.axiosConfig);
    this.pending = {};
    this.reqNum = 0;
    this.timeId = null;
    this.toastFn = toastFn;
    this.loadingFn = loadingFn;
    this.strategyModes = { ...strategyModes(this.toastFn), ...strategy };
    this.strategyModes.default = defaultStratege;
    this.initRequestInterceptors();
    this.initResponseInterceptors();
  }
  getInstance() {
    return this.instance;
  }
  beforeRequest() {
    this.reqNum++;
    clearTimeout(this.timeId);
    this.timeId = setTimeout(() => {
      this.loadingFn(true);
    }, 1300);
  }
  afterRequest() {
    this.reqNum--;
    if (this.reqNum <= 0) {
      this.clearRequest();
      this.loadingFn(false);
    }
  }
  clearRequest() {
    clearTimeout(this.timeId);
  }
  setStrategy(strategy = {}, cover = false) {
    this.strategyModes = cover ? strategy : { ...this.strategyModes, ...strategy };
  }
  setLoadingFn(fn = () => {}) {
    this.loadingFn = fn;
  }
  setToastFn(fn = () => {}) {
    this.toastFn = fn;
  }
  initRequestInterceptors() {
    const self = this;
    this.instance.interceptors.request.use(
      config => {
        !config.hideLoading && self.beforeRequest(config.url || '');
        // 发起请求时，取消掉当前正在进行的相同请求
        if (self.pending[config.url]) {
          self.pending[config.url]('取消重复请求');
        }
        config.cancelToken = new self.CancelToken(c => (self.pending[config.url] = c));
        // 使用默认响应处理策略
        config.strategy = config.strategy || self.axiosConfig.strategy;
        if (self.strategyModes[config.strategy].request) {
          config = self.strategyModes[config.strategy].request(config);
        }
        return config;
      },
      error => {
        return [null, error];
      }
    );
  }
  initResponseInterceptors() {
    const self = this;
    this.instance.interceptors.response.use(
      response => {
        if (response.config.url) {
          !response.config.hideLoading && self.afterRequest();
          delete self.pending[response.config.url || ''];
        }
        return self.strategyModes[response.config.strategy].response(response);
      },
      err => {
        self.afterRequest();

        let message = '';
        let showToast = true;

        if (err.message === 'Network Error' && !err.response) {
          // 网络异常: 错误域名，
          message = '服务不可用';
        } else if (err.response && err.response.status) {
          message =
            ERR_MESSAGE_MAP.status[err.response.status] || `未知异常码: ${err.response.status}`;
        } else if (err.message === '取消重复请求') {
          message = '取消重复请求';
          showToast = false;
        }
        showToast && this.toastFn(message || '服务异常，请稍后重试');
        const error = new Error(message);
        console.error(error);
        return [null, error];
      }
    );
  }
}

export default HttpRequest;
