import axios from 'axios';
import localStorage from './localStorage';
import Bus from './bus';

let CancelToken = axios.CancelToken; // axios中断取消请求
let httpQueue = []; // 请求队列
let cancelArr = []; // 存储取消方法
let timeout = 200; // 延时loading
let loadingTimeOut;
let closeTimeOut;

const config = {
  timeout: 60000,
};
const instance = axios.create(Object.assign({}, config));

// 挂载扩展配置
instance._extend = {
  notice: '',
  modal: '', // modal组件
  loginUrl: '',
  refresTokenHost: '',
  dataParseMode: 'default'
};

let isRefreshing = true;
let subscriber = [];
const addSubscriber = callback => subscriber.push(callback);
const wait = async seconds => new Promise(resolve => setTimeout(resolve, seconds));
const onAccessTokenFetched = () => {
  subscriber.forEach(callback => callback());
  subscriber = [];
};

const refreshToken = async () => {
  return instance.get(`${instance._extend.refresTokenHost}/oauth/refreshtoken?refreshtoken=${localStorage.get('refreshToken')}`);
};

/*
 * 刷新token逻辑
 */
const refreshRequest = async options => {
  const promise = new Promise(resolve => addSubscriber(() => resolve(instance.request(options))));
  if (isRefreshing) {
    isRefreshing = false;
    await wait(500);
    const data = await refreshToken();
    localStorage.set('token', data.accessToken);
    onAccessTokenFetched();
    isRefreshing = true;
  }
  return promise;
};

function beforeRequest(model, isLoading = true) {
  if (!isLoading) return;
  if (!model) model = 'common';
  httpQueue.push(model);
  loadingTimeOut = setTimeout(() => {
    if (httpQueue.length !== 0) {
      Bus.$emit(`${model}ShowLoading`);
    }
  }, timeout);
  closeTimeOut = setTimeout(() => {
    Bus.$emit(`${model}HideLoading`);
  }, 30 * 60 * 1000);
}
function afterRequest(model, isLoading = true) {
  if (!isLoading) return;
  if (!model) model = 'common';
  clearTimeout(loadingTimeOut);
  const index = httpQueue.indexOf(model);
  if (index > -1) {
    httpQueue.splice(index, 1);
  }
  if (httpQueue.indexOf(model) === -1) {
    clearTimeout(closeTimeOut);
    setTimeout(() => {
      Bus.$emit(`${model}HideLoading`);
    }, 100);
  }
}
function clearRequest() {
  let modelArr = Array.from(new Set(httpQueue));
  for (const model of modelArr) {
    Bus.$emit(`${model}HideLoading`);
  }
  httpQueue = [];
}

const showLoginResultPopups = () => {
  window.modalPopup = instance._extend.modal;
  instance._extend.modal.info({
    title: '温馨提示',
    content: '登录完成',
    okText: '关闭',
  });
};

const openWindow = url => {
  let a = document.createElement('a');
  document.body.appendChild(a);
  a.addEventListener('click', function(){
    o = window.open();
    o.focus();
    showLoginResultPopups();
    o.location.href = url;
  });
  a.click();
  document.body.removeChild(a);
  return false;
};

const showLoginPopups = () => {
  if (instance._extend.modal) {
    let href = window.location.href;
    href = href.replace(/\?token=[a-z0-9\-A-Z]+/g, '');
    href = instance._extend.loginUrl + '?url=' + window.btoa(href);
    instance._extend.modal.confirm({
      title: '会话过期',
      content: '当前会话已过期，请重新登录。',
      okText: '登录',
      cancelText: '关闭',
      onOk: function() {
        instance._extend.modal.remove();
        openWindow(href);
        setTimeout(() => {
          showLoginResultPopups();
        }, 1000);
        return false;
      },
    });
  } else {
    window.location.href = instance._extend.loginUrl + '?url=' + window.btoa(href);
  }
};

// http request 拦截器
instance.interceptors.request.use(
  config => {
    config.timeout = 60000;
    beforeRequest(config.headers._loading, config.headers._isLoading);
    const token = localStorage.get('token');
    const user = localStorage.get('user');
    const tenantId =localStorage.get('tenantId'); 
    if (user) {
      config.headers['Account'] = user.account;
      config.headers['Account-Name'] = encodeURI(user.name);
    }
    if (token) {
      config.headers['X-Auth-Token'] = token;
      config.headers['Access-Token'] = token;
      config.headers['X-Requested-With'] = 'XMLHttpRequest';
    }
    if(tenantId){
      config.headers['qg-tenant-id'] = tenantId;
    }

    let rewriteMap = {};

    let isUrl = /^(http:\/\/|https:\/\/)/;
    if (isUrl.test(config.url)) {
      let arr = config.url.split('/');
      let path = (arr && arr[3]) || '';
      if (rewriteMap[path]) {
        let hasQuery = config.url.indexOf('?') !== -1;
        config.url += !hasQuery ? '?' : '&';
      }
    }

    //普通的get请求
    // TODO: 某些接口不支持operator参数的话，需要排除
    if (config.method === 'get' && config.url.indexOf('offline_record') !== -1) {
      config.data = true;
      config.headers.common['content-type'] = 'application/x-www-form-urlencoded';
    }
    return config;
  },
  error => {
    return Promise.reject(error);
  }
);

instance.interceptors.response.use(
  response => {
    afterRequest(response.config.headers._loading, response.config.headers._isLoading);
    if (response.headers['content-type'] === 'application/octet-stream' || (response.headers['content-type'] === 'application/vnd.ms-excel;charset=UTF-8' && response.data.byteLength !== 'undefied')) {
      return response.data;
    }
    if (['blob', 'arraybuffer'].includes(response.config.responseType)) {
      return response.data;
    }

    // http error
    // 200 - 300 之间的都是正常响应
    if (response.status > 200 || response.status < 300) {
      const codeArr = [0, '0', 2000];
      const code = response.data && response.data.code;
      if (response.data && ((code === '0000' && response.data.businessCode === '0000') || codeArr.includes(code))) {
        if (instance._extend.dataParseMode === 'NOT_PARSE') {
          return response.data;
        }

        if (instance._extend.dataParseMode === 'PARSE_BODY' && code === 0) {
          return response.data.body;
        }

        return response.data.data || response.data;
      }
      if (+code === 4033) {
        // token过期;
        return refreshRequest(response.config);
      }
      if (+code === 4010 || +code === 4011) {
        showLoginPopups();
        return;
      }
      instance._extend.notice.error({
        desc: (response && response.data && (response.data.detail || response.data.msg)) || '后端服务异常',
        // desc: ''
      });
      return Promise.reject(response && response.data);
    }
    // eg: file response does not have data
    instance._extend.notice.error({
      desc: (response && response.data && (response.data.detail || response.data.msg)) || '后端服务异常',
    });
    return Promise.reject(response.data);
  },
  error => {
    if (axios.isCancel(error) || error.message.indexOf('Network') > -1) {
      clearRequest();
      return Promise.reject(error);
    } else {
      afterRequest(error.response && error.response.config.headers._loading);
      if (error.response) {
        switch (error.response.status) {
          case 401:
            // todo: 系统中有3处处理401的地方
            showLoginPopups();
            return;
        }
      }
    }

    instance._extend.notice.error({
      title: error.message || '后端服务异常',
    });
    return Promise.reject(error);
  }
);

const http = {
  _extend(options) {
    Object.assign(instance._extend, options);
  },
  get(url, options) {
    return new Promise((resolve, reject) => {
      instance
        .get(
          url,
          Object.assign({}, options, {
            cancelToken: new CancelToken(c => {
              cancelArr.push(c);
            }),
          })
        )
        .then(res => {
          resolve(res);
        })
        .catch(e => {
          if (!axios.isCancel(e)) {
            reject(e);
          }
        });
    });
  },
  post(url, data, options) {
    return new Promise((resolve, reject) => {
      instance
        .post(
          url,
          data,
          Object.assign({}, options, {
            cancelToken: new CancelToken(c => {
              cancelArr.push(c);
            }),
          })
        )
        .then(res => {
          resolve(res);
        })
        .catch(e => {
          reject(e);
        });
    });
  },
  put(url, data, options) {
    return new Promise((resolve, reject) => {
      instance
        .put(
          url,
          data,
          Object.assign({}, options, {
            cancelToken: new CancelToken(c => {
              cancelArr.push(c);
            }),
          })
        )
        .then(res => {
          resolve(res);
        })
        .catch(e => {
          if (!axios.isCancel(e)) {
            reject(e);
          }
        });
    });
  },
  delete(url, options) {
    return new Promise((resolve, reject) => {
      instance
        .delete(
          url,
          Object.assign({}, options, {
            cancelToken: new CancelToken(c => {
              cancelArr.push(c);
            }),
          })
        )
        .then(res => {
          resolve(res);
        })
        .catch(e => {
          if (!axios.isCancel(e)) {
            reject(e);
          }
        });
    });
  },
  cancel() {
    cancelArr.forEach(cancel => cancel());
  },
};

export default http;
