import axios from 'axios';
import localStorage from './localStorage';
import Bus from './bus';

let CancelToken = axios.CancelToken; // axios中断取消请求
let httpQueue = []; // 请求队列
let cancelArr = []; // 存储取消方法
let timeout = 200; // 延时loading
let loadingTimeOut;
let closeTimeOut;

const config = {
  timeout: 30000,
};
const instance = axios.create(Object.assign({}, config));

// 挂载扩展配置
instance._extend = {
  notice: '',
  loginUrl: '',
  refresTokenHost: '',
};

let isRefreshing = true;
let subscriber = [];
const addSubscriber = callback => subscriber.push(callback);
const wait = async seconds => new Promise(resolve => setTimeout(resolve, seconds));
const onAccessTokenFetched = () => {
  subscriber.forEach(callback => callback());
  subscriber = [];
};

const refreshtoken = async () => {
  return instance.get(`${instance._extend.refresTokenHost}/oauth/refreshtoken?refreshtoken=${localStorage.get('refreshToken')}`);
};

/*
 * 刷新token逻辑
 */
const refreshRequest = async options => {
  const promise = new Promise(resolve => addSubscriber(() => resolve(instance.request(options))));
  if (isRefreshing) {
    isRefreshing = false;
    await wait(500);
    const data = await refreshtoken();
    localStorage.set('token', data.accessToken);
    onAccessTokenFetched();
    isRefreshing = true;
  }
  return promise;
};

function beforeRequest(model) {
  if (!model) model = 'common';
  httpQueue.push(model);
  loadingTimeOut = setTimeout(() => {
    if (httpQueue.length !== 0) {
      Bus.$emit(`${model}ShowLoading`);
    }
  }, timeout);
  closeTimeOut = setTimeout(() => {
    Bus.$emit(`${model}HideLoading`);
  }, 30 * 60 * 1000);
}
function afterRequest(model) {
  if (!model) model = 'common';
  clearTimeout(loadingTimeOut);
  var index = httpQueue.indexOf(model);
  if (index > -1) {
    httpQueue.splice(index, 1);
  }
  if (httpQueue.indexOf(model) === -1) {
    clearTimeout(closeTimeOut);
    setTimeout(() => {
      Bus.$emit(`${model}HideLoading`);
    }, 100);
  }
}
function clearRequest() {
  let modelArr = Array.from(new Set(httpQueue));
  for (let i = 0; i < modelArr.length; i++) {
    Bus.$emit(`${modelArr[i]}HideLoading`);
  }
  httpQueue = [];
}
// http request 拦截器
instance.interceptors.request.use(
  config => {
    beforeRequest(config.headers._loading);
    const token = localStorage.get('token');
    const user = localStorage.get('user');
    if (user) {
      config.headers['Account'] = user.account;
      config.headers['Account-Name'] = encodeURI(user.name);
    }
    if (token) {
      config.headers['X-Auth-Token'] = token;
      config.headers['Access-Token'] = token;
      config.headers['X-Requested-With'] = 'XMLHttpRequest';
    }

    let rewriteMap = {};

    let isUrl = /^(http:\/\/|https:\/\/)/;
    if (isUrl.test(config.url)) {
      let arr = config.url.split('/');
      let path = (arr && arr[3]) || '';
      if (rewriteMap[path]) {
        let hasQuery = config.url.indexOf('?') !== -1;
        config.url += !hasQuery ? '?' : '&';
      }
    }

    // todo: 某些接口不支持operator参数的话，需要排除
    if (config.method === 'post' || config.method === 'get') {
      if (typeof config.data === 'object') {
      } else if (typeof config.data === 'string' && config.method === 'post') {
      } else if (config.method === 'get') {
        // //普通的get请求
        if (config.url.indexOf('offline_record') !== -1) {
          config.data = true;
          config.headers.common['content-type'] = 'application/x-www-form-urlencoded';
        }
      }
    }
    return config;
  },
  error => {
    return Promise.reject(error);
  }
);

instance.interceptors.response.use(
  response => {
    afterRequest(response.config.headers._loading);
    if (response.headers['content-type'] === 'application/octet-stream' || (response.headers['content-type'] === 'application/vnd.ms-excel;charset=UTF-8' && response.data.byteLength !== 'undefied')) {
      return response.data;
    }
    if (response.config.responseType === 'arraybuffer') {
      return response.data;
    }

    // http error
    // 200 - 300 之间的都是正常响应
    if (response.status > 200 || response.status < 300) {
      const codeArr = [0, '0', 2000];
      if (response.data && ((response.data.code === '0000' && response.data.businessCode === '0000') || codeArr.includes(response.data.code))) {
        return response.data.data || response.data;
      }
      if (response.data.code === 4033) {
        // token过期;
        return refreshRequest(response.config);
      }
      if (response.data.code === 4010 || response.data.code === 4011) {
        var href = window.location.href;
        href = href.replace(/\?token=[a-z0-9\-A-Z]+/g, '');
        window.location.href = instance._extend.loginUrl + '?url=' + window.btoa(href);
        return;
      }
      instance._extend.notice.error({
        desc: (response && response.data && (response.data.detail || response.data.msg)) || '后端服务异常',
        // desc: ''
      });
      return Promise.reject(response && response.data);
    }
    // eg: file response does not have data
    instance._extend.notice.error({
      desc: (response && response.data && (response.data.detail || response.data.msg)) || '后端服务异常',
    });
    return Promise.reject(response.data);
  },
  error => {
    instance._extend.notice.error({
      title: '后端服务异常',
    });
    if (axios.isCancel(error) || error.message.indexOf('Network') > -1) {
      clearRequest();
    } else {
      afterRequest(error.response && error.response.config.headers._loading);
      if (error.response) {
        switch (error.response.status) {
          case 401:
            // todo: 系统中有3处处理401的地方
            var href = window.location.href;
            href = href.replace(/\?token=[a-z0-9\-A-Z]+/g, '');
            window.location.href = instance._extend.loginUrl + '?url=' + window.btoa(href);
            return;
        }
      }
    }
    return Promise.reject(error);
  }
);

const http = {
  _extend(options) {
    Object.assign(instance._extend, options);
  },
  get(url, options) {
    return new Promise((resolve, reject) => {
      instance
        .get(
          url,
          Object.assign({}, options, {
            cancelToken: new CancelToken(c => {
              cancelArr.push(c);
            }),
          })
        )
        .then(res => {
          resolve(res);
        })
        .catch(e => {
          if (!axios.isCancel(e)) {
            reject(e);
          }
        });
    });
  },
  post(url, data, options) {
    return new Promise((resolve, reject) => {
      instance
        .post(
          url,
          data,
          Object.assign({}, options, {
            cancelToken: new CancelToken(c => {
              cancelArr.push(c);
            }),
          })
        )
        .then(res => {
          resolve(res);
        })
        .catch(e => {
          reject(e);
        });
    });
  },
  put(url, data, options) {
    return new Promise((resolve, reject) => {
      instance
        .put(
          url,
          data,
          Object.assign({}, options, {
            cancelToken: new CancelToken(c => {
              cancelArr.push(c);
            }),
          })
        )
        .then(res => {
          resolve(res);
        })
        .catch(e => {
          if (!axios.isCancel(e)) {
            reject(e);
          }
        });
    });
  },
  delete(url, options) {
    return new Promise((resolve, reject) => {
      instance
        .delete(
          url,
          Object.assign({}, options, {
            cancelToken: new CancelToken(c => {
              cancelArr.push(c);
            }),
          })
        )
        .then(res => {
          resolve(res);
        })
        .catch(e => {
          if (!axios.isCancel(e)) {
            reject(e);
          }
        });
    });
  },
  cancel() {
    cancelArr.map(cancel => cancel());
  },
};

export default http;
