'use strict'
const request = require('request')
const debug = require('debug')('request-proxy')

const Agent = require('agentkeepalive');
const HttpsAgent = require('agentkeepalive').HttpsAgent;

module.exports = function (options) {
  const optionsDefault = {
    agent: {
      keepAlive: true,
      // 当 socket 超过 60 秒都没有任何活动，就会被当作超时处理掉
      timeout: 60000,
      // 每个域名最大socket数
      maxSockets: 800, // 默认Infinity
      // 最大空闲 socket 数
      maxFreeSockets: 10, // 默认256
      // 空闲的 KeepAlive socket 最长可以存活30秒
      freeSocketKeepAliveTimeout: 30000
    },
    timeout: 15000
  }
  options = Object.assign({}, optionsDefault, options)

  /**
   * time: 调试请求各阶段时间信息
   * followRedirect: 是否重定向
   * json: body自动解析为json对象
   */
  const curlDefault = {
    followRedirect: false,
    json: true,
    time: false,
    timeout: options.timeout
  }

  /**
   * resRes: 是否返回解压后的response
   * throw：error发生时是否抛出异常被 错误中间件 自动捕获。否则把错误返回给代码
   */
  const extraDefault = {
    rawRes: false,
    throw: true
  }

  const pipeDefault = {
    time: false,
    timeout: options.timeout
  }

  const isProduction = process.env.NODE_ENV === 'production'
  const rewriteUrl = function (source, to) {
    return source.replace(/\/\/([^\/]+)\//, `//${to}/`).replace(/_ip_=[\w\.:%]+&?/, '')
  }

  const httpAgent = new Agent(options.agent);
  const httpsAgent = new HttpsAgent();

  return async function (ctx, next) {
    if (ctx.pipe || ctx.curl) return next()

    ctx.req.connection.setNoDelay(true)
    ctx.curl = function (option, extra) {
      if (typeof option === 'string') {
        option = { uri: option }
      }

      if (!option.method) {
        option.method = ctx.method
      }

      option = Object.assign({}, curlDefault, option)
      extra = Object.assign({}, extraDefault, extra)

      if (!isProduction && ctx.request.query['_ip_']) {
        option.uri = rewriteUrl(option.uri, ctx.request.query._ip_)
      }

      debug(option)

      // if (!option.agent && !/^https/.test(option.uri)) {
      option.agent = (option && option.uri && option.uri.indexOf('https') != -1) ? httpsAgent : httpAgent
      // } else if (!option.agent && /^https/.test(option.uri)) {
      //   option.agent = httpsAgent
      // }

      return new Promise(function (resolve, reject) {
        request(option, function (err, response, body) {
          if (err) {
            if (extra.throw) {
              reject(err)
            } else {
              resolve(err)
            }
          } else {
            resolve(extra.rawRes ? response : {
              status: response.statusCode,
              headers: response.headers,
              body: body || {}
            })
          }
        })
      })
    }

    ctx.pipe = function (option) {
      if (typeof option === 'string') {
        option = { uri: option }
      }
      option = Object.assign({}, pipeDefault, option)

      if (!isProduction && ctx.request.query['_ip_']) {
        option.uri = rewriteUrl(option.uri, ctx.request.query._ip_)
      }

      if (!option.method) {
        option.method = ctx.method
      }

      if (!option.agent && !/^https/.test(option.uri)) {
        option.agent = httpAgent
      } else if (!option.agent && /^https/.test(option.uri)) {
        option.agent = httpsAgent
      }

      let send = function () {
        return request(option, function (err, response) {
          if (option.time && options.logger) {
            options.logger.info(option.uri, JSON.stringify(response && response.timings || ''))
          }
        })
      }

      if (ctx.is('json', 'urlencoded') && ctx.request.body) {
        if (ctx.is('urlencoded')) {
          option.form = ctx.request.rawBody
        } else {
          option.body = ctx.request.rawBody
        }

        option.headers = Object.assign({}, ctx.request.headers)
        delete option.headers['host']

        debug(option)
        return new Promise(function (resolve, reject) {
          let r = send()
          r.on('error', function (err) {
            console.error(err)
            reject(err);
          })
            .on('response', function () {
              ctx.respond = false
              r.pipe(ctx.res)
              resolve({})
            })
        })
      }


      return new Promise(function (resolve, reject) {
        let r = ctx.req.pipe(send())
        r.on('error', function (err) {
          console.error(err)
          reject(err);
        })
          .on('response', function () {
            ctx.respond = false
            r.pipe(ctx.res)
            resolve({})
          })
      })
    }

    return next()
  }
}