'use strict'
const request = require('request')
const debug = require('debug')('request-proxy')

const Agent = require('agentkeepalive');
const HttpsAgent = require('agentkeepalive').HttpsAgent;

module.exports = function (options) {
  const optionsDefault = {
    agent: {
      keepAlive: true,
      // 当 socket 超过 60 秒都没有任何活动，就会被当作超时处理掉
      timeout: 60000,
      // 每个域名最大socket数
      maxSockets: 800, // 默认Infinity
      // 最大空闲 socket 数
      maxFreeSockets: 10, // 默认256
      // 空闲的 KeepAlive socket 最长可以存活30秒
      freeSocketKeepAliveTimeout: 30000
    },
    timeout: 15000
  }
  options = Object.assign({}, optionsDefault, options)

  /**
   * time: 调试请求各阶段时间信息
   * followRedirect: 是否重定向
   * json: body自动解析为json对象
   */
  const curlDefault = {
    followRedirect: false,
    json: true,
    timeout: options.timeout
  }

  /**
   * resRes: 是否返回解压后的response
   * throw：error发生时是否抛出异常被 错误中间件 自动捕获。否则把错误返回给代码
   */
  const extraDefault = {
    rawRes: false,
    throw: true
  }

  const pipeDefault = {
    timeout: options.timeout
  }

  const httpAgent = new Agent(options.agent);
  const httpsAgent = new HttpsAgent();

  const logger = (option, response) => {
    if (options.logger) {
      options.logger.info(JSON.stringify({ api: option.uri, ...(response && response.timings)}))
    }
  }

  return async function (ctx, next) {
    if (ctx.pipe || ctx.curl) return next()

    ctx.req.connection.setNoDelay(true)
    ctx.curl = function (option, extra) {
      if (typeof option === 'string') {
        option = {uri: option}
      }

      if (!option.method) {
        option.method = ctx.method
      }

      option = Object.assign({}, curlDefault, option)
      extra = Object.assign({}, extraDefault, extra)

      debug(option)

      if (!option.agent) {
        option.agent = option.uri.indexOf('https') === 0 ? httpsAgent : httpAgent
      }

      return new Promise(function (resolve, reject) {
        request(option, function(err, response, body) {
          if (err) {
            if (extra.throw) {
              reject(err)
            } else {
              resolve(err)
            }
          } else {
            logger(option, response)
            resolve(extra.rawRes ? response : {
              status: response.statusCode,
              headers: response.headers,
              body: body || {}
            })
          }
        })
      })
    }

    ctx.pipe = function (option) {
      if (typeof option === 'string') {
        option = {uri: option}
      }
      option = Object.assign({}, pipeDefault, option)

      if (!option.method) {
        option.method = ctx.method
      }

      debug(option)

      if (!option.agent) {
        option.agent = option.uri.indexOf('https') === 0 ? httpsAgent : httpAgent
      }

      return new Promise(function (resolve, reject) {
        let r = ctx.req.pipe(request(option, function (err, response) {
          logger(option, response)
        }))
        r.on('error', function (err) {
            reject(err);
          })
          .on('response', function () {
            ctx.respond = false
            r.pipe(ctx.res)
            resolve({})
          })
      })
    }

    return next()
  }
}