import {Component, Mixins, Prop, Watch} from 'vue-property-decorator';
import { Getter, State } from 'vuex-class';
import { reduce, ceil, subtract, divide } from 'lodash';
import { ContextMenu } from '@editor/mixins/contextMenu.mixin';
import Upload from './component/Upload/index.vue';
import ColorSelector from './component/ColorSelector/index.vue';
import BaseSelect from './component/BaseSelect/index.vue';
import Textarea from './component/Textarea/index.vue';
import Number from './component/Number/index.vue';
import { resizeDiv, getStyle } from '@/service/utils.service';

@Component({ components: { Upload, ColorSelector, BaseSelect, Textarea, Number }, name: 'DynamicForm' })
export default class DynamicForm extends Mixins(ContextMenu) {
  @State(state => state.editor.curEleIndex) curEleIndex;
  @State(state => state.editor.curChildIndex) curChildIndex;
  @Getter('pageData') pageData;

  form: object = {};

  get curElement() {
    let element = {};
    if (this.curEleIndex !== null) {
      if (this.curChildIndex !== null && this.pageData.elements[this.curEleIndex]) {
        element = this.pageData.elements[this.curEleIndex].child[this.curChildIndex];
      } else {
        element = this.pageData.elements[this.curEleIndex];
      }
    }
    return element;
  }

  get point() {
    return this.curEleIndex || this.curEleIndex === 0 ? this.pageData.elements[this.curEleIndex]?.point : { h: 0, w: 0 };
  }

  get commonStyle() {
    return (this.curEleIndex || this.curEleIndex === 0) && (this.curChildIndex || this.curChildIndex === 0) ? this.pageData.elements[this.curEleIndex].child[this.curChildIndex].commonStyle : { h: 0, w: 0 };
  }

  @Watch('curElement', { immediate: true, deep: true })
  onElementChange(newVal) {
    newVal?.schame?.forEach(schame => {
      this.$set(this.form, schame.key, newVal.props[schame.key]);
    });
  }

  @Watch('form', { immediate: true, deep: true })
  onFormChange(newVal) {
    this.$emit('modProps', this.form);
  }

  get eleName() {
    let result = '';
    if (!this.curChildIndex && this.curChildIndex !== 0) {
      result = this.pageData?.elements[this.curEleIndex]?.name ?? '';
    }
    return result;
  }

  resizedEvent(h, w, responsive) {
    const elements = this.pageData.elements[this.curEleIndex];
    if (responsive) {
      resizeDiv(this.form.backgroundImage, 667, 375, (imgHeight) => {
        this.updatePageInfo({ containerIndex: this.curEleIndex, data: { ...elements, point: { ...elements.point, w: w ?? elements.point.w, h: imgHeight ?? elements.point.h, responsive: true } } });
      });
    } else {
      this.updatePageInfo({ containerIndex: this.curEleIndex, data: { ...elements, point: { ...elements.point, w: w ?? elements.point.w, h: h ?? elements.point.h } } });
    }
  }

  resizedChildEvent(type) {
    this.$emit('resizedChildEvent', type);
    // const containerEle = this.$refs.freedomContainer[this.curEleIndex];
  }

  changeAlignType(type) {
    const freedomBody = document.querySelector('.freedom-body');
    const curElement = (freedomBody as Element).children[this.curChildIndex];
    const [ containerW, containerH ] = [getStyle(freedomBody, 'width'), getStyle(freedomBody, 'height')];
    const [ eleW, eleH ] = [getStyle(curElement, 'width'), getStyle(curElement, 'height')];
    const elements = this.pageData.elements[this.curEleIndex].child[this.curChildIndex];
    let { left, top } = elements.commonStyle;
    switch (type) {
    case 'top':
      top = 0; break;
    case 'right':
      left = subtract(containerW, eleW); break;
    case 'bottom':
      top = subtract(containerH, eleH); break;
    case 'left':
      left = 0; break;
    case 'vertical':
      top = ceil(divide(subtract(containerH, eleH), 2), 2); break;
    case 'horizontal':
      left = ceil(divide(subtract(containerW, eleW), 2), 2); break;
    default: break;
    }
    this.updatePageInfo({ containerIndex: this.curEleIndex, childIndex: this.curChildIndex, data: { ...elements, commonStyle: { ...elements.commonStyle, left, top } } });
  }

  getComponent(type) {
    let result = 'Input';
    switch (type) {
    case 'text':
      result = 'Input';
      break;
    case 'select':
      result = 'BaseSelect';
      break;
    case 'checkbox' :
      result = 'Checkbox';
      break;
    case 'ColorSelector' :
      result = 'ColorSelector';
      break;
    case 'textarea' :
      result = 'Textarea';
      break;
    case 'number' :
      result = 'Number';
      break;
    case 'Upload':
      result = 'Upload';
      break;
    }
    return result;
  }
}