import { kebabCase, maxBy } from 'lodash';
import { Getter, Action, State, Mutation } from 'vuex-class';
import { Mixins, Component, Watch } from 'vue-property-decorator';
import DynamicComponent from '@editor/component/DynamicComponent/index.vue';
import VueGridLayout from 'vue-grid-layout';
import FreedomContainer from '../../component/FreedomContainer/index.vue';
import DynamicForm from '../../component/DynamicForm/index.vue';
import components from '@qg/cherry-ui/src/index.js';
import LoginForm from '@/lib/Form/index.vue';
import { ContextMenu } from '@editor/mixins/contextMenu.mixin';

import BasicPageForm from '@editor/component/BasicPageForm/index.vue';

import { basicComponents, businessComponents } from '@/lib/config';

@Component({components: { DynamicComponent, FreedomContainer, DynamicForm, GridLayout: VueGridLayout.GridLayout,
  GridItem: VueGridLayout.GridItem, LoginForm, BasicPageForm }, name: 'DashBoard'})
export default class DashBoard extends Mixins(ContextMenu) {
  @Mutation('ADD_ELEMENTS') addElements;
  @Mutation('SET_CUR_ELE_INDEX') setCurEleIndex;
  @Mutation('SET_CUR_CHILD_INDEX') setCurChildIndex;
  @Mutation('SET_PAGE_INFO') setPageInfo;
  @Mutation('SET_PAGE_DATA') setPageData;
  @Action('resetPageData') resetPageData;
  @Action('savePageData') savePageData;
  @Action('getPageDate') getPageDate;
  @Action('getTemplateList') getTemplateList;
  @Action('setTemplateInfo') setTemplateInfo;
  @Getter('pageId') pageId;
  @Getter('pageData') pageData;
  @State(state => state.editor.draggable) draggable;
  @State(state => state.editor.curEleIndex) curEleIndex;
  @State(state => state.editor.curChildIndex) curChildIndex;
  @State(state => state.editor.templateList) templateList;

  activeName: string = '1';
  isCollapsed: boolean = true;
  isDragIn: boolean = false;
  isDraging: boolean = false;
  resources: object = { basicComponents, businessComponents };
  showSubmitPopup: boolean = false;

  async created() {
    const { pageId, templateId } = this.$route.params;
    this.resetPageData();
    if (pageId) {
      this.getPageDate({ pageId });
    } else if (templateId) {
      this.setTemplateInfo({ pageId: templateId });
    }
    this.getTemplateList();
  }

  get layout() {
    return this.pageData.elements.map(v => v.point);
  }

  get curElement() {
    let element = {};
    if (this.curEleIndex !== null) {
      if (this.curChildIndex !== null) {
        element = this.pageData.elements[this.curEleIndex].child[this.curChildIndex];
      } else {
        element = this.pageData.elements[this.curEleIndex];
      }
    }
    return element;
  }

  // 选择组件库
  selectMaterial(val: string) {
    this.activeName = val;
  }

  async save(type, pageConfig) {
    if (!type) {
      this.showSubmitPopup = true;
    } else {
      this.pageData.elements.sort((a, b) => a.point.y - b.point.y);
      const { pageName, pageDescribe, coverImage, showDownload, isPublish, isTemplate } = pageConfig;
      const pageInfo = { page: JSON.stringify(this.pageData), author: 'congmin.hao', isPublish,  pageName, pageDescribe, coverImage, showDownload, isTemplate };
      if (+this.pageId) { pageInfo.id = this.pageId; }
      await this.savePageData(pageInfo);
      this.showSubmitPopup = false;
      if (type === 'preview') {
        window.open(`http://localhost:7001/activity/${this.pageId}`);
      }
    }
  }

  exit() {
    this.$router.back();
  }

  handleElementClick(curEleIndex = null, curChildIndex = null) {
    this.toggle(false);
    this.setCurEleIndex(curEleIndex);
    this.setCurChildIndex(curChildIndex);
  }

  toggle(val) {
    if (val) {
      this.setCurEleIndex(null);
      this.setCurChildIndex(null);
    }
    this.isCollapsed = val;
  }

  dragstart() {
    this.isDraging = true;
  }

  dragend() {
    this.isDraging = false;
  }

  dragenter() {
    console.log('dragenter');
    this.isDragIn = true;
  }

  dragleave() {
    console.log('dragleave');
    this.isDragIn = false;
  }

  dragover(event) {
    if (event.target.classList.contains('freedom')) {
      event.dataTransfer.dropEffect = 'move';
    } else {
      event.dataTransfer.dropEffect = 'copy';
    }
  }

  modProps(props) {
    console.log('modProps');
    if (this.curEleIndex !== null) {
      if (this.curChildIndex !== null) {
        this.updatePageInfo({ containerIndex: this.curEleIndex, childIndex: this.curChildIndex, data: { ...this.pageData.elements[this.curEleIndex].child[this.curChildIndex], props } });
      } else {
        this.updatePageInfo({ containerIndex: this.curEleIndex, data: { ...this.pageData.elements[this.curEleIndex], props } });
      }
    }
  }

  drops(event) {
    this.isDragIn = false;
    this.isCollapsed = false;
    const data = JSON.parse(event.dataTransfer.getData('text'));
    const { layerX: left, layerY: top } = event;
    event.dataTransfer.clearData();
    // template
    if (data.template) {
      this.setPageData(JSON.parse(data.template));
      this.handleElementClick(null, null);
    // freedom
    } else if (event.target.classList.contains('freedom')) {
      const { y: curY } = this.pageData.elements[event.target.dataset.index].point;
      const scrollTop = this.layout.reduce((pre, cur) => {
        if (cur.y < curY) {
          return pre + cur.h * 55.09;
        }
        return pre;
      }, 0);
      this.addElements({ containerIndex: event.target.dataset.index, data: { ...data, commonStyle: { position: 'absolute', left, top: top - scrollTop } } });
      this.handleElementClick(+event.target.dataset.index, this.pageData.elements[event.target.dataset.index].child.length - 1);
    // component
    } else {
      const { i } = maxBy(this.layout, 'i') || {};
      const y = Math.floor(top / 55.09);
      console.log('drops', i);
      this.addElements({ data: {...data, point: { ...data.point, i: i || i === 0 ? String(+i + 1) : '0', y } });
      this.handleElementClick(this.pageData.elements.length - 1, null);
    }
  }

  getProps(eleName) {
    const props = {};
    for (const key of Object.keys(components)) {
      const component = components[key];
      if (kebabCase(component.name) === eleName && component.props) {
        console.log(key, component.props);
        for (const prop of Object.keys(component.props)) {
          props[prop] = [Object, Array].includes(component.props[prop].type) ? component.props[prop].default() : component.props[prop].default;
        }
      }
    }

    return props;
  }

  resizedEvent(i, h, w) {
    const index = this.pageData.elements.findIndex(ele => ele.point.i === i);
    this.updatePageInfo({ containerIndex: index, data: { ...this.pageData.elements[index], point: { ...this.pageData.elements[index].point, w, h } } });
  }
}