import { kebabCase, maxBy } from 'lodash';
import { Getter, Action, State, Mutation } from 'vuex-class';
import { Mixins, Component, Watch } from 'vue-property-decorator';
import DynamicComponent from '@editor/component/DynamicComponent/index.vue';
import VueGridLayout from 'vue-grid-layout';
import FreedomContainer from '../../component/FreedomContainer/index.vue';
import DynamicForm from '../../component/DynamicForm/index.vue';
import components from '@qg/cherry-ui/src/index.js';
import LoginForm from '@/lib/Form/index.vue';
import DownloadGuide from '@/lib/DownloadGuide/index.vue';
import { ContextMenu } from '@editor/mixins/contextMenu.mixin';
import BasicPageForm from '@editor/component/BasicPageForm/index.vue';
import { basicComponents, businessComponents } from '@/lib/config';
import config from '@/config/index';
import localStorage from '@/service/localStorage.service';

@Component({components: { DynamicComponent, FreedomContainer, DynamicForm, GridLayout: VueGridLayout.GridLayout,
  GridItem: VueGridLayout.GridItem, LoginForm, DownloadGuide, BasicPageForm }, name: 'DashBoard'})
export default class DashBoard extends Mixins(ContextMenu) {
  @Mutation('ADD_ELEMENTS') addElements;
  @Mutation('SET_CUR_ELE_INDEX') setCurEleIndex;
  @Mutation('SET_CUR_CHILD_INDEX') setCurChildIndex;
  @Mutation('SET_PAGE_INFO') setPageInfo;
  @Mutation('SET_PAGE_DATA') setPageData;
  @Action('resetPageData') resetPageData;
  @Action('savePageData') savePageData;
  @Action('getPageDate') getPageDate;
  @Action('getTemplateList') getTemplateList;
  @Action('setTemplateInfo') setTemplateInfo;
  @Getter('pageId') pageId;
  @Getter('pageData') pageData;
  @State(state => state.editor.gridLayout.draggable) draggable;
  @State(state => state.editor.gridLayout.rowHeight) rowHeight;
  @State(state => state.editor.curEleIndex) curEleIndex;
  @State(state => state.editor.curChildIndex) curChildIndex;
  @State(state => state.editor.templateList) templateList;

  activeName: string = '1';
  isCollapsed: boolean = true;
  isDragIn: boolean = false;
  isDraging: boolean = false;
  resources: object = { basicComponents, businessComponents };
  showSubmitPopup: boolean = false;

  async created() {
    const { pageId, templateId } = this.$route.params;
    console.log(pageId, templateId);
    console.log('env', process.env);
    this.resetPageData();
    if (pageId) {
      this.getPageDate({ pageId });
    } else if (templateId) {
      this.setTemplateInfo({ pageId: templateId });
    }
    this.getTemplateList();
  }

  get layout() {
    return this.pageData.elements.map(v => v.point);
  }

  get curElement() {
    let element = {};
    if (this.curEleIndex !== null) {
      if (this.curChildIndex !== null) {
        element = this.pageData.elements[this.curEleIndex].child[this.curChildIndex];
      } else {
        element = this.pageData.elements[this.curEleIndex];
      }
    }
    return element;
  }

  // 选择组件库
  selectMaterial(val: string) {
    this.activeName = val;
  }

  async save(type, pageConfig) {
    const user = localStorage.get('user');
    if (!type) {
      this.showSubmitPopup = true;
    } else {
      this.pageData.elements.sort((a, b) => a.point.y - b.point.y);
      const { pageName, pageDescribe, coverImage, isPublish, isTemplate } = pageConfig;
      const pageInfo = { page: JSON.stringify(this.pageData), author: user?.account, isPublish,  pageName, pageDescribe, coverImage, isTemplate };
      if (+this.pageId) { pageInfo.id = this.pageId; }
      await this.savePageData(pageInfo);
      this.showSubmitPopup = false;
      if (type === 'preview') {
        window.open(`${config.apiHost}activity/${this.pageId}`);
      }
    }
  }

  exit() {
    this.$router.back();
  }

  handleElementClick(curEleIndex = null, curChildIndex = null) {
    this.toggle(false);
    this.setCurEleIndex(curEleIndex);
    this.setCurChildIndex(curChildIndex);
  }

  toggle(val) {
    if (val) {
      this.setCurEleIndex(null);
      this.setCurChildIndex(null);
    }
    this.isCollapsed = val;
  }

  dragstart() {
    this.isDraging = true;
  }

  dragend() {
    this.isDraging = false;
  }

  dragenter() {
    console.log('dragenter');
    this.isDragIn = true;
  }

  dragleave() {
    console.log('dragleave');
    this.isDragIn = false;
  }

  dragover(event) {
    if (event.target.classList.contains('freedom')) {
      event.dataTransfer.dropEffect = 'move';
    } else {
      event.dataTransfer.dropEffect = 'copy';
    }
  }

  modProps(props) {
    let currentEle = {};
    if (this.curEleIndex !== null) {
      if (this.curChildIndex !== null) {
        currentEle = this.pageData.elements[this.curEleIndex].child[this.curChildIndex];
        this.updatePageInfo({ containerIndex: this.curEleIndex, childIndex: this.curChildIndex, data: { ...currentEle, props: { ...currentEle.props, ...props } } });
      } else {
        currentEle = this.pageData.elements[this.curEleIndex];
        this.updatePageInfo({ containerIndex: this.curEleIndex, data: { ...this.pageData.elements[this.curEleIndex], props: { ...currentEle.props, ...props } } });
      }
    }
  }

  drops(event) {
    this.isDragIn = false;
    this.isCollapsed = false;
    const data = JSON.parse(event.dataTransfer.getData('text'));
    const { layerX: left, layerY: top } = event;
    event.dataTransfer.clearData();
    // template
    if (data.template) {
      this.setPageData(JSON.parse(data.template));
      this.handleElementClick(null, null);
    // freedom
    } else if (event.target.classList.contains('freedom')) {
      const { y: curY } = this.pageData.elements[event.target.dataset.index].point;
      const scrollTop = this.layout.reduce((pre, cur) => {
        if (cur.y < curY) {
          return pre + cur.h * this.rowHeight;
        }
        return pre;
      }, 0);
      this.addElements({ containerIndex: event.target.dataset.index, data: { ...data, commonStyle: { position: 'absolute', left, top: top - scrollTop } } });
      this.handleElementClick(+event.target.dataset.index, this.pageData.elements[event.target.dataset.index].child.length - 1);
    // component
    } else {
      const { i } = maxBy(this.layout, 'i') || {};
      const y = Math.floor(top / this.rowHeight);
      console.log('drops', i);
      this.addElements({ data: {...data, point: { ...data.point, i: i || i === 0 ? String(+i + 1) : '0', y } });
      this.handleElementClick(this.pageData.elements.length - 1, null);
    }
  }

  resizedEvent(i, h, w) {
    const index = this.pageData.elements.findIndex(ele => ele.point.i === i);
    this.updatePageInfo({ containerIndex: index, data: { ...this.pageData.elements[index], point: { ...this.pageData.elements[index].point, w, h } } });
  }
}