import api from '@/api/editor.api';
import { Module, GetterTree, ActionTree, MutationTree } from 'vuex';
import { cloneDeep } from 'lodash';
import {
  SET_PAGE_INFO,
  SET_DRAGABLE,
  COPY_OR_DELETE_PAGE_INFO,
  UPDATE_PAGE_INFO,
  ADD_ELEMENTS，
  SET_CUR_ELE_INDEX,
  SET_CUR_CHILD_INDEX,
  RESET_PAGE_DATA,
  SET_TEMPLATE_LIST,
  SET_PAGE_DATA,
} from './type';

import RootState from '../../state';
import EditorState, { PageInfo, defaultState, Page, PageElement } from './state';

export default class EditorModule implements Module<EditorState, RootState> {
  state: EditorState;

  getters: GetterTree<EditorState, RootState> = {
    pageData(state) {
      return state.pageInfo.page;
    },
    pageId(state) {
      return state.pageInfo.id;
    },
    pageInfo(state) {
      return state.pageInfo;
    }
  };

  actions: ActionTree<EditorState, RootState> = {
    async savePageData({ commit }, condition) {
      if (condition.id) {
        await api.updatePage(condition);
        commit(SET_PAGE_INFO, { ...condition, page: JSON.parse(condition.page as string) });
      } else {
        const res = await api.savePage(condition);
        const { page, ...rest } = res as PageInfo;
        commit(SET_PAGE_INFO, { ...rest, page: JSON.parse(page as string) });
      }
    },
    async getPageDate({ commit }, condition) {
      const res = await api.getPageById(condition);
      const { page, ...rest } = res as PageInfo;
      commit(SET_PAGE_INFO, { ...rest, page: JSON.parse(page as string) });
    },
    async getTemplateList({ commit }) {
      const res = await api.getTemplateList({ type: 'template' });
      const handledData = res.map((v, idx) => ({ eleName: `template${idx}`, title: v.pageName, coverImage: v.coverImage, page: v.page }));
      // todo 解析数据
      commit(SET_TEMPLATE_LIST, handledData);
    },
    async setTemplateInfo({commit}, condition) {
      const res = await api.getPageById(condition);
      const { page } = res as PageInfo;
      const { pageInfo } = cloneDeep(defaultState);
      commit(SET_PAGE_INFO, { ...pageInfo, page: JSON.parse(page as string) });
    },
    setDragable({ commit }, condition) {
      commit(SET_DRAGABLE, condition);
    },
    resetPageData({ commit }, condition) {
      const { pageInfo, curEleIndex, curChildIndex } = cloneDeep(defaultState);
      commit(SET_PAGE_INFO, pageInfo);
      commit(SET_CUR_ELE_INDEX, curEleIndex);
      commit(SET_CUR_CHILD_INDEX, curChildIndex);
    }
  };

  mutations: MutationTree<EditorState> = {
    [SET_PAGE_INFO](state, data) {
      state.pageInfo = data;
    },
    [SET_DRAGABLE](state, data) {
      state.gridLayout.draggable = data;
    },
    [SET_CUR_ELE_INDEX](state, curEleIndex) {
      state.curEleIndex = curEleIndex;
    },
    [SET_CUR_CHILD_INDEX](state, curChildIndex) {
      state.curChildIndex = curChildIndex;
    },
    [SET_TEMPLATE_LIST](state, data) {
      state.templateList = data;
    },
    [SET_PAGE_DATA](state, data) {
      state.pageInfo.page = data;
    },
    [COPY_OR_DELETE_PAGE_INFO](state, { type, containerIndex, childIndex }) {
      const page = (state.pageInfo.page as Page).elements;
      if (type === 'delete') {
        if (childIndex || childIndex === 0) {
          page[containerIndex].child.splice(childIndex, 1);
        } else {
          page.splice(containerIndex, 1);
        }
      } else if (type === 'copy') {
        let eleCopyed = {} as PageElement;
        if (childIndex || childIndex === 0) {
          eleCopyed = page[containerIndex].child[childIndex];
          const { left, top } = eleCopyed.commonStyle;
          page[containerIndex].child.push({ ...eleCopyed, commonStyle: { ...eleCopyed.commonStyle, left: left + 10, top: top + 10 } });
        } else {
          eleCopyed = page[containerIndex];
          page.push({ ...eleCopyed, point: { ...eleCopyed.point, i: page.length }});
        }
      }
    },
    [UPDATE_PAGE_INFO](state, { containerIndex, childIndex, data }) {
      const page = (state.pageInfo.page as Page).elements;
      if (childIndex || childIndex === 0) {
        page[containerIndex].child.splice(childIndex, 1, data);
      } else {
        page.splice(containerIndex, 1, data);
      }
    },
    [ADD_ELEMENTS](state, { containerIndex, data }) {
      const page = (state.pageInfo.page as Page).elements;
      if (containerIndex || containerIndex === 0) {
        page[containerIndex].child.push(data);
      } else {
        page.push(data);
      }
    },
  };

  constructor(initState: EditorState = cloneDeep(defaultState)) {
    this.state = initState;
  }
}