import { Controller, Context } from 'egg';
import { omitBy } from 'lodash';

type GetWhere = {
  id?: number | unknown;
  tenantId?: number | unknown;
};

export default class TenantAuthController extends Controller {

  public async add(ctx: Context) {
    const authInfo = ctx.request.body;
    await ctx.model.TenantAuth.create({ ...authInfo });
    ctx.body = ctx.helper.ok();
  }

  public async update(ctx: Context) {
    const authInfo = ctx.request.body;
    if (!authInfo.tenantId) {
      ctx.body = ctx.helper.fail({ message: '无租户id', code: '0001', businessCode: '0002' });
      return;
    }
    await ctx.model.TenantAuth.update(authInfo, {where: { id: authInfo.id }});
    ctx.body = ctx.helper.ok();
  }

  public async get(ctx: Context) {
    const where: GetWhere = {};
    if (ctx.headers['qg-tenant-id']) where.tenantId = ctx.headers['qg-tenant-id'];
    if (ctx.query.id) where.tenantId = ctx.query.id;
    try {
      const result = await ctx.model.TenantAuth.findOne({ where });
      ctx.body = ctx.helper.ok(result);
    } catch (error) {
      console.log(error);
      ctx.body = ctx.helper.fail({ message: error, code: '0001', businessCode: '0001' });
    }
    
  }

  public async getList(ctx: Context) {
    const { pageSize, pageNo, tenantName, tenantId } = ctx.query;
    const { like } = ctx.model.Sequelize.Op;
    let where = omitBy({
      tenantName: tenantName && { like: `%${tenantName}%`},
      tenantId: tenantId && { like: `%${tenantId}%`},
      enable: 1
    }, v => !v);
    const { count: total, rows: data } = await ctx.model.PageInfo.findAndCountAll({ where, limit: +pageSize || 10,
      offset: (+pageNo - 1) * +pageSize || 0,  order: [['updated_at', 'DESC']] });
    ctx.body = ctx.helper.ok({ total, data });
  }

  public async delete(ctx: Context) {
    const authInfo = await ctx.model.PageInfo.update({ enable: 0 }, {where: { id: +ctx.params.id }});
    ctx.body = ctx.helper.ok(authInfo);
  }
}