import { kebabCase, maxBy } from 'lodash';
import { Getter, Action, State, Mutation } from 'vuex-class';
import { Mixins, Component, Watch, Provide } from 'vue-property-decorator';
import DynamicComponent from '@editor/component/DynamicComponent/index.vue';
import VueGridLayout from 'vue-grid-layout';
import FreedomContainer from '../../component/FreedomContainer/index.vue';
import DynamicForm from '../../component/DynamicForm/index.vue';
import DynamicPageForm from '../../component/DynamicPageForm/index.vue';
import { basicComponents, businessComponents } from '@/lib/config';
import ContextMenuMixin from '@editor/mixins/contextMenu.mixin';
import GoodsTabsMixin from '@editor/mixins/goodsTabs.mixin';
import TransformStyleMixin from '@/page/mixins/transformStyle.mixin';
import BasicPageForm from '@editor/component/BasicPageForm/index.vue';
import config from '@/config/index';
import localStorage from '@/service/localStorage.service';
import EventBus from '@/service/eventBus.service';
// import CsLoginForm from '@/lib/Form/index.vue';
// import CsDownloadGuide from '@/lib/DownloadGuide/index.vue';
// import CsGoodsTabs from '@/lib/GoodsTabs/index.vue';
// import CsGoods from '@/lib/Goods/index.vue';
// import CsAdvertisement from '@/lib/Advertisement/index.vue';
// import CsCoupon from '@/lib/Coupon/index.vue';
// import CsPlaceholder from '@/lib/Placeholder/index.vue';
// import CsGuideCube from '@/lib/GuideCube/index.vue';
// import CsMarquee from '@/lib/Marquee/index.vue';
// CsLoginForm, CsDownloadGuide, CsGuideCube, CsGoodsTabs, CsCoupon, CsMarquee, CsAdvertisement, CsGoods, CsPlaceholder
@Component({components: { DynamicComponent, FreedomContainer, DynamicForm, GridLayout: VueGridLayout.GridLayout,
  GridItem: VueGridLayout.GridItem, BasicPageForm, DynamicPageForm }, name: 'DashBoard'})
export default class DashBoard extends Mixins(ContextMenuMixin, GoodsTabsMixin, TransformStyleMixin) {
  @Mutation('ADD_ELEMENTS') addElements;
  @Mutation('SET_CUR_ELE_INDEX') setCurEleIndex;
  @Mutation('SET_CUR_CHILD_INDEX') setCurChildIndex;
  @Mutation('SET_PAGE_INFO') setPageInfo;
  @Mutation('SET_PAGE_DATA') setPageData;
  @Mutation('UPDATE_PAGE_STYLE') setPageStyle;
  @Mutation('UPDATE_PAGE_PROPS') setPageProps;
  @Action('resetPageData') resetPageData;
  @Action('savePageData') savePageData;
  @Action('getPageDate') getPageDate;
  @Action('getTemplateList') getTemplateList;
  @Action('setTemplateInfo') setTemplateInfo;
  @Getter('pageId') pageId;
  @Getter('pageData') pageData;
  @State(state => state.editor.gridLayout.draggable) draggable;
  @State(state => state.editor.gridLayout.rowHeight) rowHeight;
  @State(state => state.editor.gridLayout.colNum) colNum;
  @State(state => state.editor.curEleIndex) curEleIndex;
  @State(state => state.editor.curChildIndex) curChildIndex;
  @State(state => state.editor.templateList) templateList;

  @Provide() editor = this;

  activeName: string = '1';
  isCollapsed: boolean = true;
  isDragIn: boolean = false;
  isDraging: boolean = false;
  resources: object = { basicComponents, businessComponents };
  showSubmitPopup: boolean = false;

  async created() {
    const { pageId, templateId } = this.$route.params;
    console.log(pageId, templateId);
    console.log('env', process.env);
    this.resetPageData();
    if (pageId) {
      await this.getPageDate({ pageId });
    } else if (templateId) {
      await this.setTemplateInfo({ pageId: templateId });
    }
    this.parseGoodsTabs();
    this.getTemplateList();
  }

  get layout() {
    return this.pageData.elements.map(v => v.point);
  }

  // 选择组件库
  selectMaterial(val: string) {
    this.activeName = val;
  }

  async save(type, pageConfig) {
    const user = localStorage.get('user');
    if (!type) {
      this.showSubmitPopup = true;
    } else {
      this.pageData.elements.sort((a, b) => a.point.y - b.point.y);
      const pageData = this.handleGoodsTabs();
      const { pageName, pageDescribe, coverImage, isPublish, isTemplate } = pageConfig;
      const pageInfo = { page: JSON.stringify(pageData), author: user?.account, isPublish,  pageName, pageDescribe, coverImage, isTemplate };
      if (+this.pageId) { pageInfo.id = this.pageId; }
      await this.savePageData({ pageInfo, pageData: this.pageData });
      this.showSubmitPopup = false;
      if (type === 'preview') {
        window.open(`${config.h5Host}activity/${this.pageId}`);
      }
    }
  }

  exit() {
    this.$router.back();
  }

  handleElementClick(curEleIndex = null, curChildIndex = null) {
    console.log('handleElementClick - DashBoard', curEleIndex, curChildIndex);
    this.toggle(false);
    this.setCurEleIndex(curEleIndex);
    this.setCurChildIndex(curChildIndex);
  }

  toggle(val) {
    this.isCollapsed = val;
  }

  dragstart() {
    this.isDraging = true;
  }

  dragend() {
    this.isDraging = false;
  }

  dragenter() {
    console.log('dragenter');
    this.isDragIn = true;
  }

  dragleave() {
    console.log('dragleave');
    this.isDragIn = false;
  }

  dragover(event) {
    if (event.target.classList.contains('freedom')) {
      event.dataTransfer.dropEffect = 'move';
    } else {
      event.dataTransfer.dropEffect = 'copy';
    }
  }

  modProps(props, ele, type) {
    if (ele === 'page') {
      if (type === 'commonStyle') {
        this.setPageStyle({ data: props });
      } else if (type === 'props') {
        this.setPageProps({ data: props });
      }
    } else if (ele === 'component') {
      let currentEle = {};
      if (this.curEleIndex !== null) {
        if (this.curChildIndex !== null) {
          currentEle = this.pageData.elements[this.curEleIndex].child[this.curChildIndex];
          this.updatePageInfo({ containerIndex: this.curEleIndex, childIndex: this.curChildIndex, data: { ...currentEle, props: { ...currentEle.props, ...props } } });
        } else {
          currentEle = this.pageData.elements[this.curEleIndex];
          this.updatePageInfo({ containerIndex: this.curEleIndex, data: { ...this.pageData.elements[this.curEleIndex], props: { ...currentEle.props, ...props } } });
        }
      }
    }
  }

  drops(event) {
    this.isDragIn = false;
    this.isCollapsed = false;
    const data = JSON.parse(event.dataTransfer.getData('text'));
    const { layerX: left, layerY: top } = event;
    event.dataTransfer.clearData();
    // template
    if (data.template) {
      this.setPageData(JSON.parse(data.template));
      this.handleElementClick(null, null);
    // freedom
    } else if (event.target.classList.contains('freedom-body')) {
      const { y: curY } = this.pageData.elements[event.target.dataset.index].point;
      const scrollTop = this.layout.reduce((pre, cur) => {
        if (cur.y < curY) {
          return pre + cur.h * this.rowHeight;
        }
        return pre;
      }, 0);
      this.addElements({ containerIndex: event.target.dataset.index, data: { ...data, commonStyle: { position: 'absolute', left, top: top - scrollTop } } });
      this.handleElementClick(+event.target.dataset.index, this.pageData.elements[event.target.dataset.index].child.length - 1);
    // component
    } else {
      const y = Math.floor(top / this.rowHeight);
      this.addElements({ data: {...data, point: { ...data.point, y } });
      this.handleElementClick(this.pageData.elements.length - 1, null);
    }
    // 调整组件高度
    this.$nextTick(() => this.adjustHeight());
  }

  resizedEvent(i, h, w) {
    const index = this.pageData.elements.findIndex(ele => ele.point.i === i);
    this.updatePageInfo({ containerIndex: index, data: { ...this.pageData.elements[index], point: { ...this.pageData.elements[index].point, w, h } } });
  }

  movedEvent(i, newX, newY) {
    EventBus.$emit('component-moved');
    console.log('MOVED i=' + i + ', X=' + newX + ', Y=' + newY);
  }

  /**
   * 调整自由容器子元素宽高及边框原点位置
   * @param {[type]} type 尺寸类型
   */
  resizedChildEvent(type) {
    const containerEle = this.$refs.container[this.curEleIndex];
    containerEle.setChildSize(type);
    containerEle.setPointStyle();
  }
}