import { Controller, Context } from 'egg';
import _, { omitBy } from 'lodash';
import { v1 as uuidv1 } from 'uuid';
export default class EditorController extends Controller {

  public async login(ctx: Context) {
    await ctx.renderClient('login.js', {});
  }

  public async home(ctx: Context) {
    const apollo = ctx.app.config.apollo || {};
    await ctx.renderClient('editor.js', { apollo });
  }

  public async save(ctx: Context) {
    const pageInfo = ctx.request.body;
    try {
      const result = await ctx.model.PageInfo.create({ ...pageInfo, uuid: uuidv1().replace(/-/g, ''), tenantId: ctx.headers['qg-tenant-id']});
      ctx.body = ctx.helper.ok(result);
      await ctx.service.redis.del(`pageInfo:${result.dataValues.uuid}`);
      await ctx.service.redis.del(`page:${result.dataValues.uuid}`);
    } catch (error) {
      if (error.message.indexOf('Data too long') > -1 && error.message.indexOf('page_data') > -1) {
        error.message = '配置组件过多，请对组件进行删减';
      }
      ctx.body = ctx.helper.fail(error);
    }
  }

  public async update(ctx: Context) {
    const pageInfo = ctx.request.body;
    if (!pageInfo.uuid) {
      pageInfo.uuid = uuidv1().replace(/-/g, '');
    }
    if (ctx.headers['qg-tenant-id']) {
      pageInfo.tenantId = +ctx.headers['qg-tenant-id'];
    }
    try {
      const result = await ctx.model.PageInfo.update(pageInfo, {where: { uuid: pageInfo.uuid, tenantId: ctx.headers['qg-tenant-id'] }});
      await ctx.service.redis.del(`pageInfo:${pageInfo.uuid}`);
      await ctx.service.redis.del(`page:${pageInfo.uuid}`);
      ctx.body = ctx.helper.ok(result);
    } catch (error) {
      if (error.message.indexOf('Data too long') > -1 && error.message.indexOf('page_data') > -1) {
        error.message = '配置组件过多，请对组件进行删减';
      }
      ctx.body = ctx.helper.fail(error);
    }
  }
  // todo 编辑器获取跳过redis，但需要注意
  public async get(ctx: Context) {
    const { isEditor } = ctx.query;
    let pageInfo = isEditor ? null : await ctx.service.redis.get(`pageInfo:${ctx.params.uuid}`);
    console.log('redis', pageInfo);
    if (!pageInfo) {
      pageInfo = await ctx.model.PageInfo.findOne({where: { uuid: ctx.params.uuid }});
      console.log('sql', pageInfo);

      if (ctx.query.lite && pageInfo) {
        // 如果存在sheme移除掉
        const page = JSON.parse(pageInfo.page || []);
        for (let i = 0; i < page.elements.length; i++) {
          delete page.elements[i].schame
        }
        if (page.scheme) {
          delete page.scheme;
        }
        pageInfo.page = JSON.stringify(page);
      }
      await ctx.service.redis.set(`pageInfo:${ctx.params.uuid}`, pageInfo);
    }
    ctx.body = ctx.helper.ok(pageInfo);
  }

  public async getList(ctx: Context) {
    const { type, pageSize, pageNo, author, pageName, isPublish, pageDescribe, uuid } = ctx.query;
    const { like } = ctx.model.Sequelize.Op;
    let where = omitBy({
      author: author && { like: `%${author}%`},
      pageName: pageName && { like: `%${pageName}%`},
      pageDescribe: pageDescribe && { like: `%${pageDescribe}%`},
      isPublish,
      uuid,
      tenantId: ctx.headers['qg-tenant-id'],
      enable: 1
    }, v => !v);
    if (type === 'list') {
      where = { ...where, isPublish: 1 };
    } else if (type === 'my') {
      where = { ...where };
    } else if (type === 'template') {
      where = { ...where, isTemplate: 1 };
    }
    const { count: total, rows: data } = await ctx.model.PageInfo.findAndCountAll({ where, limit: +pageSize || 10,
      offset: (+pageNo - 1) * +pageSize || 0,  order: [['updated_at', 'DESC']] });
    ctx.body = ctx.helper.ok({ total, data });
  }

  public async getTemplateList(ctx: Context) {
    const list = await ctx.model.PageInfo.findAll({ where: { isTemplate: 1, tenantId: ctx.headers['qg-tenant-id'] } });
    ctx.body = ctx.helper.ok(list);
  }

  public async clearCache(ctx: Context) {
    const where = {
      updated_at: {
        [ctx.model.Sequelize.Op.lt]: Date.now(),
        [ctx.model.Sequelize.Op.gt]: new Date(Date.now() - 1000 * 60 * 60 * 24 * 30)
      },
      tenantId: ctx.headers['qg-tenant-id'],
      enable: 1,
      isPublish: 1
    };
    const pageInfo = await ctx.model.PageInfo.findAll({
      attributes: ['uuid'],
      where
    });
    for (const item of pageInfo) {
      await ctx.service.redis.del(`page:${item.uuid}`);
      await ctx.service.redis.del(`pageInfo:${item.uuid}`);
      ctx.logger.info(`del redis page key page:${item.uuid}`);
    }
    ctx.body = ctx.helper.ok();
  }

  public async delete(ctx: Context) {
    const pageInfo = await ctx.model.PageInfo.update({ enable: 0 }, {where: { id: +ctx.params.pageId, tenantId: ctx.headers['qg-tenant-id'] }});
    ctx.body = ctx.helper.ok(pageInfo);
  }
}