import { Vue, Component, Watch, Provide, Mixins } from 'vue-property-decorator';
import { Getter, State, Mutation } from 'vuex-class';
import FreedomContainer from '../../component/FreedomContainer/index.vue';
import PageBottomTip from '../../component/PageBottomTip/index.vue';
import GridLayout from '../../component/VueGridLayout/GridLayout.vue';
import GridItem from '../../component/VueGridLayout/GridItem.vue';
import TransformStyleMixin from '@/page/mixins/transformStyle.mixin';
import SaMixin from '@/page/mixins/sa.mixin';
import { getStyle, debounce, isApp } from '@/service/utils.service';
import { EventBus } from '@qg/citrus-ui/src/helper/service/eventBus';
import { setAppTitleColor } from '@/service/color.service';
import localStorage from '@/service/localStorage.service';
@Component({ components: { FreedomContainer, GridLayout, GridItem, PageBottomTip }, name: 'Activity'})
export default class Activity extends Mixins(TransformStyleMixin, SaMixin) {
  @Getter('pageData') pageData;
  @State(state => state.activity.pageInfo.pageName) pageName;
  @State(state => state.activity.pageInfo.appLoginState) appLoginState;
  @State(state => state.activity.gridLayout.rowHeight) rowHeight;
  @Mutation('SET_PAGE_ELEMENTS') setPageElement;
  @Mutation('UPDATE_APP_LOGIN_STATE') updateAppLoginState;

  @Provide() editor = this;
  isEditor: boolean = false;
  bottomInfo: object = {
    x: 0,
    y: 0,
    w: 375,
    h: 60,
    i: 'page-bottom-top'
  };
  isLayoutComReady = false;
  showBackTop = false;

  targetEle: HTMLElement | null = null;
  loading: boolean = true;
  modfiTabsStyleDebounce = debounce(this.modfiTabsStyle, 300);

  get layout() {
    return this.pageData && this.pageData.elements.map(v => v.point) || [];
  }

  get backTopList() {
    return isApp && this.pageData && this.pageData.props.btAttachVal ? this.pageData.props.btAttachVal : [];
  }

  @Watch('pageData', { deep: true })
  onPageDataChange(val) {
    const lastGridItem = val.elements[val.elements.length - 1];
    const lastGridItemPoint = lastGridItem.point;
    if (lastGridItemPoint && lastGridItemPoint.w) {
      this.bottomInfo = {
        x: 0,
        y: 0,
        w: 375,
        h: 60,
        i: 'page-bottom-top'
      };
      if (lastGridItemPoint.w > this.bottomInfo.w) { this.bottomInfo.w = lastGridItemPoint.w; }
      this.bottomInfo.y = this.bottomInfo.y + lastGridItemPoint.y + lastGridItemPoint.h;
    }
    if (isApp) {
      // 当页面数据改变，并且渲染完成后改变app 标题栏颜色
      this.$nextTick(() => {
        debounce(setAppTitleColor(this.pageData.props.titleBgColor), 300);
      });
    }
  }

  @Watch('pageName', { immediate: true })
  onPageNameChange(newVal) {
    if (EASY_ENV_IS_BROWSER && newVal) {
      document.title = newVal;
      // 如果是 iOS 设备，则使用如下 hack 的写法实现页面标题的更新
      if (navigator.userAgent.match(/\(i[^;]+;( U;)? CPU.+Mac OS X/)) {
        const hackIframe = document.createElement('iframe');
        hackIframe.style.display = 'none';
        hackIframe.src = '/public/asset/fixIosTitle.html?r=' + Math.random();
        document.body.appendChild(hackIframe);
        setTimeout(() => {
          document.body.removeChild(hackIframe);
        }, 300);
      }
      // 适配
      this.modifyPoints();
    }
  }

  mounted() {
    this.targetEle = document.querySelector('body');
    this.showBackTop = true;
    this.pageVisibilityChange();
    if (EASY_ENV_IS_BROWSER) {
      EventBus.$on('NATIVE_EVENT_LOGIN', json => {
        console.log('xyqbNativeEvent got', json);
        if (json.event === 'loginAndLogoutNotification') {
          const appData = json.data || {};
          if (appData && appData.login) {
            localStorage.set('vccToken', appData.token);
            this.updateAppLoginState(true);
          } else {
            this.updateAppLoginState(false);
            localStorage.remove('vccToken');
          }
        }
      });
    }
  }
  fetchApi(options) {
    const { store, route } = options;
    const { pageId } = route.params;
    return store.dispatch('getPageDate', { pageId });
  }

  modifyPoints() {
    const clientWidth = document.documentElement.clientWidth > 768 ? 375 : document.documentElement.clientWidth;
    const elements = this.pageData?.elements?.map(v => {
      const point = {
        ...v.point,
        h: v.point.h * this.rowHeight * clientWidth / 375,
        w: clientWidth
      };
      return { ...v, point };
    });
    this.setPageElement(elements);
  }

  dot(title) {
    console.log(title, '点击了');
  }

  layoutReadyEvent(newLayout) {
    this.$nextTick(() => {
      console.log('layoutReadyEvent');
      const loadingEle = document.querySelector('.mainload');
      console.log('loadingEle', loadingEle);
      if (!loadingEle) { return; }
      loadingEle.style.display = 'none';
    });
  }

  layoutUpdatedEvent() {
    console.log('layoutUpdatedEvent');
    // 高度更新时重新计算导航组件样式
    this.modfiTabsStyleDebounce();
  }
  hideBottomBtns(state) {
    this.showBackTop = state;
  }
  pageVisibilityChange() {
    if (EASY_ENV_IS_NODE) { return; }
    let hidden = 'hidden';
    let visibilityChange = 'visibilitychange';
    if (typeof document.hidden !== 'undefined') {
      hidden = 'hidden';
      visibilityChange = 'visibilitychange';
    } else if (typeof document.webkitHidden !== 'undefined') {
      hidden = 'webkitHidden';
      visibilityChange = 'webkitvisibilitychange';
    }
    window.addEventListener(visibilityChange, () => {
      this.handleChange(document[hidden]);
    }, false);
  }
  handleChange(state) {
    if (!state) {
      if (isApp) {
        setAppTitleColor(this.pageData.props.titleBgColor);
      }
    }
  }
  modfiTabsStyle() {
    const tabsEle = document.querySelector('.tabs');
    // console.log('tabsEle', tabsEle);
    if (tabsEle) {
      const gridItemEle = tabsEle?.parentNode;
      // console.log('gridItemEle', gridItemEle, gridItemEle?.classList);
      if (gridItemEle?.classList.contains('vue-grid-item')) {
        // 处理transform
        if (gridItemEle.classList.contains('del_transform')) {
          gridItemEle.classList.remove('del_transform');
        }
        const transform = getStyle(gridItemEle, 'transform');
        if (transform && transform.length) {
          const transformY = transform.split('(')[1].split(')')[0].split(',')[5];
          gridItemEle.style.top = `${transformY}px`;
          gridItemEle.className += ' del_transform';
        }
        // 处理backgroundColor
        const activityEle = document.querySelector('.activity');
        const activityBgColor = getStyle(activityEle, 'backgroundColor');
        const gridItemEleBgColor = getStyle(gridItemEle, 'backgroundColor');
        const gridItemEleBgImg = getStyle(gridItemEle, 'backgroundImage');
        const gridItemEleBgRepeat = getStyle(gridItemEle, 'backgroundRepeat') || 'no-repeat';
        const gridItemEleBgSize = getStyle(gridItemEle, 'backgroundSize') || 'auto';
        const crTabs = tabsEle.childNodes[0];
        crTabs.style.backgroundColor = activityBgColor;
        const stickyEle = crTabs?.childNodes[0];
        if (stickyEle?.classList.contains('cr-sticky') && stickyEle?.childNodes) {
          stickyEle.childNodes[0]?.style.backgroundColor = gridItemEleBgColor === 'rgba(0, 0, 0, 0)' ? activityBgColor : gridItemEleBgColor;
          // polyfill backgroundImage
          if (gridItemEleBgImg) {
            stickyEle.childNodes[0]?.style.backgroundImage = gridItemEleBgImg;
            stickyEle.childNodes[0]?.style.backgroundSize = gridItemEleBgSize;
            stickyEle.childNodes[0]?.style.backgroundRepeat = gridItemEleBgRepeat;
          }
        }
      }
    }
  }
}