import { Vue, Component, Watch, Provide, Mixins } from 'vue-property-decorator';
import { Getter, State, Mutation } from 'vuex-class';
import FreedomContainer from '../../component/FreedomContainer/index.vue';
import BackTop from '../../component/BackTop/index.vue';
import PageBottomTip from '../../component/PageBottomTip/index.vue';
import GridLayout from '../../component/VueGridLayout/GridLayout.vue';
import GridItem from '../../component/VueGridLayout/GridItem.vue';
import TransformStyleMixin from '@/page/mixins/transformStyle.mixin';
import SaMixin from '@/page/mixins/sa.mixin';
import { getStyle, debounce, isApp, isWxMp } from '@/service/utils.service';
import { setAppTitleColor } from '@/service/color.service';
@Component({ components: { FreedomContainer, GridLayout, GridItem, PageBottomTip, BackTop }, name: 'Activity'})
export default class Activity extends Mixins(TransformStyleMixin, SaMixin) {
  @Getter('pageData') pageData;
  @State(state => state.activity.pageInfo.pageName) pageName;
  @State(state => state.activity.gridLayout.rowHeight) rowHeight;
  @Mutation('SET_PAGE_ELEMENTS') setPageElement;

  @Provide() editor = this;
  isEditor: boolean = false;
  bottomInfo: object = {
    x: 0,
    y: 0,
    w: 375,
    h: 60,
    i: 'page-bottom-top'
  };
  isLayoutComReady = false;
  showBackTop = false;
  targetEle: HTMLElement | null = null;
  loading: boolean = true;
  modfiTabsStyleDebounce = debounce(this.modfiTabsStyle, 300);

  get layout() {
    if (!isApp && !isWxMp) {
      this.pageData.elements = this.pageData.elements.filter(v => v.name !== 'cs-search-bar');
    }
    return this.pageData && this.pageData.elements.map(v => v.point) || [];
  }

  @Watch('pageData', { deep: true })
  onPageDataChange(val) {
    const lastGridItem = val.elements[val.elements.length - 1];
    const lastGridItemPoint = lastGridItem?.point;
    if (lastGridItemPoint && lastGridItemPoint.w) {
      this.bottomInfo = {
        x: 0,
        y: 0,
        w: 375,
        h: 60,
        i: 'page-bottom-top'
      };
      if (lastGridItemPoint.w > this.bottomInfo.w) { this.bottomInfo.w = lastGridItemPoint.w; }
      this.bottomInfo.y = this.bottomInfo.y + lastGridItemPoint.y + lastGridItemPoint.h;
    }
  }

  @Watch('pageName', { immediate: true })
  onPageNameChange(newVal) {
    if (EASY_ENV_IS_BROWSER && newVal) {
      document.title = newVal;
      // 如果是 iOS 设备，则使用如下 hack 的写法实现页面标题的更新
      if (navigator.userAgent.match(/\(i[^;]+;( U;)? CPU.+Mac OS X/)) {
        const hackIframe = document.createElement('iframe');
        hackIframe.style.display = 'none';
        hackIframe.src = '/public/asset/fixIosTitle.html?r=' + Math.random();
        document.body.appendChild(hackIframe);
        setTimeout(() => {
          document.body.removeChild(hackIframe);
        }, 300);
      }
      // 适配
      this.modifyPoints();
    }
  }
  mounted() {
    this.targetEle = document.querySelector('body');
    this.showBackTop = true;
  }
  fetchApi(options) {
    const { store, route } = options;
    const { pageId } = route.params;
    return store.dispatch('getPageDate', { pageId });
  }

  modifyPoints() {
    const clientWidth = document.documentElement.clientWidth > 768 ? 375 : document.documentElement.clientWidth;
    const elements = this.pageData?.elements?.map(v => {
      const point = {
        ...v.point,
        h: v.point.h * this.rowHeight * clientWidth / 375,
        w: clientWidth
      };
      return { ...v, point };
    });
    this.setPageElement(elements);
  }

  dot(title) {
    console.log(title, '点击了');
  }

  layoutReadyEvent(newLayout) {
    this.$nextTick(() => {
      const loadingEle = document.querySelector('.mainload');
      console.log('loadingEle', loadingEle);
      if (!loadingEle) { return; }
      loadingEle.style.display = 'none';
      if (isApp) {
        setAppTitleColor(this.pageData.props.titleBgColor);
      }
    });
  }
  getPageElements(ids = []) {
    return ids.length === 0 ? this.pageData.elements : ids.map(item => {
      const targetEl =
      this.pageData.elements.find(it => it.id === (item.componentId || item));
      return targetEl;
    });
  }
  setGridItemTransform(index, status) {
    console.log('setGridItemTransform', index, status);
    const elements = JSON.parse(JSON.stringify(this.pageData?.elements));
    elements[index].noTransforms = status;
    this.setPageElement(elements);
  }
  layoutUpdatedEvent() {
    console.log('layoutUpdatedEvent');
    // 高度更新时重新计算导航组件样式
    this.modfiTabsStyleDebounce();
  }
  // overlay遮罩显示时隐藏悬浮窗
  hideBottomBtns(state) {
    this.showBackTop = state;
  }
  // 计算商品导航组件位置
  modfiTabsStyle() {
    const tabsEle = document.querySelector('.tabs');
    // console.log('tabsEle', tabsEle);
    if (tabsEle) {
      const gridItemEle = tabsEle?.parentNode;
      // console.log('gridItemEle', gridItemEle, gridItemEle?.classList);
      if (gridItemEle?.classList.contains('vue-grid-item')) {
        // 处理transform
        if (gridItemEle.classList.contains('del_transform')) {
          gridItemEle.classList.remove('del_transform');
        }
        const transform = getStyle(gridItemEle, 'transform');
        if (transform && transform.length && transform !== 'none') {
          const transformY = transform.split('(')[1].split(')')[0].split(',')[5];
          gridItemEle.style.top = `${transformY}px`;
          gridItemEle.className += ' del_transform';
        }
        // 处理backgroundColor
        const activityEle = document.querySelector('.activity');
        const activityBgColor = getStyle(activityEle, 'backgroundColor');
        const gridItemEleBgColor = getStyle(gridItemEle, 'backgroundColor');
        const gridItemEleBgImg = getStyle(gridItemEle, 'backgroundImage');
        const gridItemEleBgRepeat = getStyle(gridItemEle, 'backgroundRepeat') || 'no-repeat';
        const gridItemEleBgSize = getStyle(gridItemEle, 'backgroundSize') || 'auto';
        const crTabs = tabsEle.childNodes[0];
        crTabs.style.backgroundColor = activityBgColor;
        const stickyEle = crTabs?.childNodes[0];
        if (stickyEle?.classList.contains('cr-sticky') && stickyEle?.childNodes) {
          stickyEle.childNodes[0]?.style.backgroundColor = gridItemEleBgColor === 'rgba(0, 0, 0, 0)' ? activityBgColor : gridItemEleBgColor;
          // polyfill backgroundImage
          if (gridItemEleBgImg) {
            stickyEle.childNodes[0]?.style.backgroundImage = gridItemEleBgImg;
            stickyEle.childNodes[0]?.style.backgroundSize = gridItemEleBgSize;
            stickyEle.childNodes[0]?.style.backgroundRepeat = gridItemEleBgRepeat;
          }
        }
      }
    }
  }
}