import { Vue, Component, Watch, Provide, Mixins } from 'vue-property-decorator';
import { Getter, State, Mutation } from 'vuex-class';
import FreedomContainer from '../../component/FreedomContainer/index.vue';
import GridLayout from '../../component/VueGridLayout/GridLayout.vue';
import GridItem from '../../component/VueGridLayout/GridItem.vue';
import TransformStyleMixin from '@/page/mixins/transformStyle.mixin';
import { getStyle } from '@/service/utils.service';

@Component({ components: { FreedomContainer, GridLayout, GridItem }, name: 'Activity'})
export default class Activity extends Mixins(TransformStyleMixin) {
  @Getter('pageData') pageData;
  @State(state => state.activity.pageInfo.pageName) pageName;
  @State(state => state.activity.gridLayout.rowHeight) rowHeight;
  @Mutation('SET_PAGE_ELEMENTS') setPageElement;

  @Provide('editor');

  isLayoutComReady = false;
  showBackTop = false;
  targetEle: HTMLElement | null = null;
  loading: boolean = true;

  get layout() {
    return this.pageData && this.pageData.elements.map(v => v.point) || [];
  }

  @Watch('pageName', { immediate: true })
  onPageNameChange(newVal) {
    if (EASY_ENV_IS_BROWSER && newVal) {
      document.title = newVal;
      // 如果是 iOS 设备，则使用如下 hack 的写法实现页面标题的更新
      if (navigator.userAgent.match(/\(i[^;]+;( U;)? CPU.+Mac OS X/)) {
        const hackIframe = document.createElement('iframe');
        hackIframe.style.display = 'none';
        hackIframe.src = '/public/asset/fixIosTitle.html?r=' + Math.random();
        document.body.appendChild(hackIframe);
        setTimeout(() => {
          document.body.removeChild(hackIframe);
        }, 300);
      }
      // 适配
      this.modifyPoints();
    }
  }

  mounted() {
    console.log('loading', this);
    this.targetEle = document.querySelector('body');
    this.showBackTop = true;
    const timer = setTimeout(() => {
      this.modfiTabsStyle();
    }, 700);
    this.$once('hook:beforeDestroy', () => {
      clearTimeout(timer);
    });
  }

  fetchApi(options) {
    const { store, route } = options;
    const { pageId } = route.params;
    return store.dispatch('getPageDate', { pageId });
  }

  modifyPoints() {
    const clientWidth = document.documentElement.clientWidth > 768 ? 375 : document.documentElement.clientWidth;
    const elements = this.pageData?.elements?.map(v => {
      const point = {
        ...v.point,
        h: v.point.h * this.rowHeight * clientWidth / 375,
        w: clientWidth
      };
      return { ...v, point };
    });
    this.setPageElement(elements);
  }

  dot(title) {
    console.log(title, '点击了');
  }

  getSaInfo(element) {
    const { id, title, name } = element;
    return {
      assembly_id: id,
      assembly_type: title,
      assembly_name: name
    };
  }

  layoutReadyEvent(newLayout) {
    // console.log('Mounted layout: ', newLayout);
    // this.loading = false;
  }

  modfiTabsStyle() {
    const tabsEle = document.querySelector('.tabs');
    if (tabsEle) {
      const gridItemEle = tabsEle?.parentNode;
      if (gridItemEle?.classList.contains('vue-grid-item')) {
        // 处理transform
        const transform = getStyle(gridItemEle, 'transform');
        const transformY = transform.split('(')[1].split(')')[0].split(',')[5];
        gridItemEle.style.transform = 'none';
        gridItemEle.style.top = `${transformY}px`;
        // 处理backgroundColor
        const activityEle = document.querySelector('.activity');
        const activityBgColor = getStyle(activityEle, 'backgroundColor');
        const gridItemEleBgColor = getStyle(gridItemEle, 'backgroundColor');
        const crTabs = tabsEle.childNodes[0];
        crTabs.style.backgroundColor = activityBgColor;
        const stickyEle = crTabs?.childNodes[0];
        if (stickyEle?.classList.contains('cr-sticky') && stickyEle?.childNodes) {
          stickyEle.childNodes[0]?.style.backgroundColor = gridItemEleBgColor === 'rgba(0, 0, 0, 0)' ? activityBgColor : gridItemEleBgColor;
        }
      }
    }
  }
}