import { isAndroid, isIOS } from './validation.service';

/**
 * hash路由获取url参数
 * @param {String} searchString  输入字符串
 * @return: {String} str 字符串
 */
export function parseSearch(searchString) {
  if (!searchString) {
    return {};
  }
  if (!searchString.includes('?')) {
    return {};
  }
  return searchString
    .split('?')[1]
    .split('#')[0]
    .split('&')
    .reduce((result, next) => {
      const pair = next.split('=');
      try {
        result[decodeURIComponent(pair[0])] = decodeURIComponent(pair[1]);
      } catch (e) {
        // eslint-disable-next-line
        //   result[decodeURIComponent(pair[0])] = window.$URL.decode(pair[1]);
      }
      return result;
    }, {});
}
/**
 * @description: 移动端监听软键盘弹起或收起，IOS，android
 * @param  {Boolean} off 是否销毁
 * @return {Function} callback 软键盘弹起或收取回调, true => 弹起， false => 隐藏
 */

export function onKeyboardStateChange(callback = () => {}, off = false) {
  if (isIOS) {
    let isReset = true; //是否归位

    const focusinHandler = () => {
      isReset = false; //聚焦时键盘弹出，焦点在输入框之间切换时，会先触发上一个输入框的失焦事件，再触发下一个输入框的聚焦事件
      callback(true);
    };

    const focusoutHandler = () => {
      isReset = true;
      setTimeout(() => {
        //当焦点在弹出层的输入框之间切换时先不归位
        if (isReset) {
          // window.scroll(0, 0); // 【暂时取掉】确定延时后没有聚焦下一元素，是由收起键盘引起的失焦，则强制让页面归位
          document.body.scrollTop = 0;
          document.documentElement.scrollTop = 0;
          callback(false);
        }
      }, 30);
    };
    if (off) {
      document.body.removeEventListener('focusin', focusinHandler);
      document.body.removeEventListener('focusout', focusoutHandler);
    } else {
      document.body.addEventListener('focusin', focusinHandler);
      document.body.addEventListener('focusout', focusoutHandler);
    }
  }

  if (isAndroid) {
    const originHeight = document.documentElement.clientHeight || document.body.clientHeight;
    const resizeHandler = () => {
      const resizeHeight = document.documentElement.clientHeight || document.body.clientHeight;
      const activeElement = document.activeElement;
      if (resizeHeight < originHeight) {
        // 键盘弹起后逻辑
        if (
          activeElement &&
          (activeElement.tagName === 'INPUT' || activeElement.tagName === 'TEXTAREA')
        ) {
          setTimeout(() => {
            activeElement.scrollIntoView({ block: 'center' }); //焦点元素滚到可视区域的问题
          }, 0);
          callback(true);
        }
      } else {
        // 键盘收起后逻辑
        callback(false);
      }
    };
    if (off) {
      window.removeEventListener('resize', resizeHandler);
    } else {
      window.addEventListener('resize', resizeHandler);
    }
  }
}

export function throttle(fn, wait) {
  let flag = true;
  return function() {
    if (flag) {
      fn.apply(this, arguments);
      flag = false;
      setTimeout(() => {
        flag = true;
      }, wait);
    }
  };
}

export function changeTitleOfIOS() {
  if (isIOS) {
    const hackIframe = document.createElement('iframe');
    hackIframe.style.display = 'none';
    hackIframe.src = '/fixIosTitle.html?r=' + Math.random();
    document.body.appendChild(hackIframe);
    setTimeout(() => {
      document.body.removeChild(hackIframe);
    }, 300);
  }
}

// 清除键盘
export function clearKeyboard() {
  var input = document.getElementsByTagName('input');
  for (let i = 0; i < input.length; i++) {
    input[i].blur();
  }
  var textarea = document.getElementsByTagName('textarea');
  for (let i = 0; i < textarea.length; i++) {
    textarea[i].blur();
  }
}
