const path = require("path");
const resolve = dir => path.join(__dirname, dir);
const SentryPlugin = require("webpack-sentry-plugin");
const CompressionPlugin = require("compression-webpack-plugin");
const SentryConfig = require("./.sentryclirc");
const IS_PROD = process.env.NODE_ENV === "production";

module.exports = {
  chainWebpack: config => {
    config.plugin("html").tap(args => {
      args[0].title = "芒果保险";
      return args;
    });
    // 修复HMR
    config.resolve.symlinks(true);

    // 移除 prefetch 插件(针对生产环境首屏请求数进行优化)
    config.plugins.delete("prefetch");

    // 移除 preload 插件(针对生产环境首屏请求数进行优化)   preload 插件的用途：https://cli.vuejs.org/zh/guide/html-and-static-assets.html#preload
    config.plugins.delete("preload");

    config.module
      .rule("images")
      .use("url-loader")
      .loader("url-loader")
      .tap(options => Object.assign(options, { limit: 10000 }));

    // 添加别名
    config.resolve.alias.set("@", resolve("src"));
    // svg-sprite
    config.module
      .rule("svg")
      .exclude.add(resolve("src/assets/icons"))
      .end();
    config.module
      .rule("icons")
      .test(/\.svg$/)
      .include.add(resolve("src/assets/icons"))
      .end()
      .use("svg-sprite-loader")
      .loader("svg-sprite-loader")
      .options({
        symbolId: "icon-[name]"
      })
      .end();
    // 组件静态资源引入
    config.module
      .rule("vue")
      .use("vue-loader")
      .loader("vue-loader")
      .tap(options => {
        return {
          ...options,
          transformAssetUrls: {
            "cr-image": "src",
            "cr-button": "icon"
          }
        };
      });
    // 分包优化
    config.optimization.splitChunks({
      chunks: "all",
      cacheGroups: {
        pdfjs: {
          name: "chunk-pdfjs", // split cherryUI into a single package
          priority: 21, // the weight needs to be larger than libs and app or it will be packaged into libs or app
          test: /[\\/]node_modules[\\/]pdfjs-dist[\\/]es5[\\/]/, // in order to adapt to cnpm
          chunks: "all",
          reuseExistingChunk: true,
          enforce: true
        },
        cherryUI: {
          name: "chunk-cherryUI", // split cherryUI into a single package
          priority: 20, // the weight needs to be larger than libs and app or it will be packaged into libs or app
          test: /[\\/]node_modules[\\/]@qg[\\/]_?cherry-ui(.*)/, // in order to adapt to cnpm
          chunks: "initial",
          reuseExistingChunk: true,
          enforce: true
        },
        saSdk: {
          name: "chunk-sa-sdk", // split cherryUI into a single package
          priority: 19, // the weight needs to be larger than libs and app or it will be packaged into libs or app
          test: /[\\/]node_modules[\\/]sa-sdk-javascript[\\/]/, // in order to adapt to cnpm
          chunks: "initial",
          reuseExistingChunk: true,
          enforce: true
        },
        libs: {
          name: "chunk-libs",
          test: /[\\/]node_modules[\\/]/,
          priority: 10,
          chunks: "initial",
          reuseExistingChunk: true,
          enforce: true
        },
        svg: {
          name: "chunk-svg",
          priority: 9,
          chunks: "initial",
          test: resolve("src/assets/icons/svg"),
          reuseExistingChunk: true,
          enforce: true
        },
        images: {
          name: "chunk-image",
          priority: 8,
          chunks: "initial",
          test: resolve("src/assets/images"),
          reuseExistingChunk: true,
          enforce: true
        },
        api: {
          name: "chunk-api",
          priority: 7,
          chunks: "all",
          minChunks: 3,
          test: resolve("src/api"),
          reuseExistingChunk: true,
          enforce: true
        },
        commons: {
          name: "chunk-commons",
          test: resolve("src"), // can customize your rules
          chunks: "all",
          minChunks: 3,
          maxInitialRequests: 5,
          minSize: 0,
          reuseExistingChunk: true,
          enforce: true
        }
      }
    });
    // https:// webpack.js.org/configuration/optimization/#optimizationruntimechunk
    config.optimization.runtimeChunk("single");
    return config;
  },
  configureWebpack: config => {
    config.plugins.push(new SentryPlugin(SentryConfig));
    if (IS_PROD) {
      config.plugins.push(
        new CompressionPlugin({
          test: /\.js$|\.html$|\.css$|\.jpg$|\.jpeg$|\.png/, // 需要压缩的文件类型
          threshold: 10240, // 归档需要进行压缩的文件大小最小值，我这个是10K以上的进行压缩
          deleteOriginalAssets: false, // 是否删除原文件
          minRatio: 0.8
        })
      );
      //生产环境取消 console.log
      // config.optimization.minimizer[0].options.terserOptions.compress.drop_console = true;
    }
  },
  lintOnSave: true,
  runtimeCompiler: false, // 是否使用包含运行时编译器的 Vue 构建版本
  productionSourceMap: IS_PROD // 生产环境的 source map
};
