import React from 'react';
import { Button, Popover } from 'antd';
import UploadImage from '@/components/UploadImg/index.jsx';

// 品牌审核状态
/**
 * @description: 品牌审核状态 1-待审核
 */
export const auditStatusWait = 1;
/**
 * @description: 品牌审核状态 2-审核通过
 */
export const auditStatusPass = 2;
/**
 * @description: 品牌审核状态 3-审核拒绝
 */
export const auditStatusReject = 3;

/**
 * @description: 品牌审核状态枚举 0-无 1-待审核，2-审核通过，3-审核拒绝
 */
export const brandStatusEnum = {
  // 0: '-',
  [auditStatusWait]: '待审核',
  [auditStatusPass]: '审核通过',
  [auditStatusReject]: '驳回',
};
/**
 * @description: 品牌审核 2审核通过 3驳回
 */
export const brandAuditEnum = {
  [auditStatusPass]: '审核通过',
  [auditStatusReject]: '驳回',
};
// 操作状态 查看、修改、添加、补充资质
/**
 * @description: 品牌操作状态 查看
 */
export const brandActionFind = 'find';
/**
 * @description: 品牌操作状态 修改
 */
export const brandActionEdit = 'edit';
/**
 * @description: 品牌操作状态 添加
 */
export const brandActionAdd = 'add';
/**
 * @description: 品牌操作状态 补充资质
 */
export const brandActionSupplement = 'supplement';

/**
 * @description: 列表基础字段
 */
export const brandBaseColumn = [
  {
    title: '品牌名称',
    dataIndex: 'name',
    key: 'name',
    align: 'center',
  },
  {
    title: '中文名称',
    dataIndex: 'chineseName',
    key: 'chineseName',
    hideInSearch: true,
    align: 'center',
  },
  {
    title: '英文名称',
    key: 'englishName',
    dataIndex: 'englishName',
    hideInSearch: true,
    align: 'center',
  },
  {
    title: '品牌缩写或别称',
    key: 'alias',
    dataIndex: 'alias',
    hideInSearch: true,
    align: 'center',
  },
];

// 品牌列表字段
export const brandColumn = config => {
  const { onAction } = config;
  return [
    ...brandBaseColumn,
    {
      title: '审核状态',
      key: 'status',
      dataIndex: 'status',
      align: 'center',
      valueEnum: brandStatusEnum,
      render: (_, r) => {
        const { status } = r;
        return status === auditStatusReject ? (
          <Popover content={r.rejectReason} title="驳回原因" trigger="hover">
            <Button type="link" danger>
              驳回
            </Button>
          </Popover>
        ) : (
          brandStatusEnum[status]
        );
      },
    },
    {
      title: '操作',
      hideInSearch: true,
      dataIndex: 'option',
      align: 'center',
      width: 200,
      render: (_, r) => [
        (r.modifiable && (
          <Button key="check" type="primary" onClick={() => onAction(r, 'edit')}>
            修改
          </Button>
        )) ||
          '',
        ([null, auditStatusReject].includes(r.status) && !r.modifiable && (
          <Button key="supplement" type="primary" ghost onClick={() => onAction(r, 'supplement')}>
            补充资质
          </Button>
        )) ||
          '',
      ],
    },
  ];
};

// 品牌信息字段
export const brandInfoColumn = config => {
  const { actionStatus } = config;
  const disabled = brandActionSupplement === actionStatus;
  const baseInfo = [
    {
      title: '资质证书',
      dataIndex: 'qualifyUrl',
      formItemProps: {
        rules: [{ required: true, message: '请选择资质证书' }],
      },
      renderFormItem: () => <UploadImage limit={1} />,
    },
    {
      title: '授权证书',
      dataIndex: 'authorizationUrl',
      formItemProps: {
        rules: [{ required: true, message: '请选择授权证书' }],
      },
      renderFormItem: () => <UploadImage limit={1} />,
    },
    {
      title: '品牌名称',
      dataIndex: 'name',
      maxLength: 50,
      fieldProps: {
        disabled,
      },
      formItemProps: {
        rules: disabled ? [] : [{ required: true, message: '请输入品牌名称' }],
      },
    },
    {
      title: '中文名称',
      dataIndex: 'chineseName',
      maxLength: 50,
      fieldProps: {
        disabled,
      },
    },
    {
      title: '英文名称',
      dataIndex: 'englishName',
      maxLength: 100,
      fieldProps: {
        disabled,
      },
    },
    {
      title: '品牌缩写或别称',
      dataIndex: 'alias',
      maxLength: 50,
      formItemProps: {
        rules: disabled ? [] : [{ required: true, message: '请输入品牌缩写或别称' }],
      },
      fieldProps: {
        disabled,
      },
    },
    {
      title: '长方形LOGO上传',
      dataIndex: 'horizontalLogo',
      formItemProps: {
        rules: disabled ? [] : [{ required: true, message: '请上传长方形LOGO上传' }],
        extra: (
          <div>
            <div>尺寸要求:219*72</div>
            <div>
              素材要求：1、透明底；2、上下左右最少留白2px，具体以保证整体大小一致，视觉平衡为准
            </div>
          </div>
        ),
      },
      fieldProps: {
        disabled,
      },
      renderFormItem: () => <UploadImage limit={1} disabled={disabled} width={219} height={72} />,
    },
    {
      title: 'LOGO上传',
      dataIndex: 'logo',
      formItemProps: {
        rules: disabled ? [] : [{ required: true, message: '请上传LOGO' }],
        extra: <span>尺寸要求:192*192</span>,
      },
      fieldProps: {
        disabled,
      },
      renderFormItem: () => <UploadImage limit={1} disabled={disabled} width={192} height={192} />,
    },
  ];
  return baseInfo;
};
