import { Button, Select, Popconfirm, notification } from 'antd';
import React, { useRef, useEffect, useState } from 'react';
import { PageHeaderWrapper } from '@ant-design/pro-layout';
import ProTable from '@ant-design/pro-table';
import lodash from 'lodash';
import { connect } from 'dva';
import { AFTER_SALE_ADDRESS } from '@/../config/permission.config';
import styles from './index.less';
import AddrForm from './form';
import { del, query, supplier } from './services';

const { Option } = Select;

const TableList = props => {
  const { permissions } = props;
  const canEditable = permissions[AFTER_SALE_ADDRESS.EDITABLE];
  const [supplierList, setSupplierList] = useState([]);
  const [shopId, setShopId] = useState(null);
  const [visible, setVisible] = useState(false);
  const [formValue, setFormValue] = useState({});
  const actionRef = useRef();

  useEffect(() => {
    async function fetchData() {
      const data = await supplier();
      setSupplierList(data);
    }
    fetchData();
  }, []);
  const changeSupplier = res => {
    setShopId(res);
  };
  const reload = type => {
    if (type) {
      setShopId(null);
    }
    actionRef.current.reload();
  };
  const delAction = async ({ id }) => {
    const data = await del(id);
    if (data.businessCode === '0000') {
      reload();
      notification.success({ message: $t('deleted.successfully') });
    } else {
      notification.error({ message: data.msg || $t('delete.failed') });
    }
  };
  const editAction = async row => {
    setFormValue(() => lodash.cloneDeep(row));
    setVisible(true);
  };
  const addAction = () => {
    setFormValue({});
    setVisible(true);
  };
  const onClose = res => {
    setVisible(false);
    if (res) {
      reload();
    }
  };
  const columns = [
    {
      title: $t('aftersales.address.name'),
      dataIndex: 'addressName',
      key: 'addressName',
      hideInSearch: true,
    },
    {
      title: $t('supplier.name'),
      dataIndex: 'shopId',
      key: 'shopId',
      hideInTable: true,
      valueType: 'select',
      renderFormItem: () => (
        <Select onChange={changeSupplier} value={shopId} allowClear>
          {supplierList.map(item => (
            <Option value={item.id} key={item.id}>
              {item.name}
            </Option>
          ))}
        </Select>
      ),
    },
    {
      title: $t('supplier.name'),
      dataIndex: 'supplierName',
      key: 'supplierName',
      hideInSearch: true,
    },
    {
      title: $t('province'),
      dataIndex: 'provinceName',
      key: 'provinceName',
      hideInSearch: true,
      width: 100,
    },
    {
      title: $t('city'),
      dataIndex: 'cityName',
      key: 'cityName',
      hideInSearch: true,
      width: 100,
    },
    {
      title: $t('county'),
      dataIndex: 'countyName',
      key: 'countyName',
      hideInSearch: true,
      width: 100,
    },
    {
      title: $t('town'),
      dataIndex: 'townName',
      key: 'townName',
      hideInSearch: true,
      width: 100,
    },
    {
      title: $t('full.address'),
      dataIndex: 'detail',
      key: 'detail',
      hideInSearch: true,
    },
    {
      title: $t('recipient.name'),
      dataIndex: 'receiverName',
      key: 'receiverName',
      hideInSearch: true,
      width: 150,
    },
    {
      title: $t('receiver.tel'),
      dataIndex: 'receiverTel',
      key: 'receiverTel',
      hideInSearch: true,
      width: 150,
    },
    {
      title: $t('remark'),
      dataIndex: 'remark',
      key: 'remark',
      hideInSearch: true,
    },
    {
      title: $t('actions'),
      dataIndex: 'option',
      key: 'option',
      valueType: 'option',
      render: (_, row) => {
        if (canEditable) {
          return [
            <Button
              key="edit"
              type="primary"
              onClick={() => {
                editAction(row);
              }}
            >{$t('edit')}</Button>,
            <Popconfirm
              placement="topLeft"
              title={$t('confirm.deletion')}
              onConfirm={() => {
                delAction(row);
              }}
              okText={$t('delete')}
              cancelText={$t('cancel')}
            >
              <Button key="del" type="danger" className={styles.btn}>{$t('delete')}</Button>
            </Popconfirm>,
          ];
        }
        return [];
      },
    },
  ];
  return (
    <PageHeaderWrapper>
      <ProTable
        className={styles.protable}
        actionRef={actionRef}
        columns={columns}
        params={{ shopId }}
        request={res => query(res)}
        rowKey={r => r.shopId}
        bordered
        scroll={{ x: 1500 }}
        toolBarRender={() => [
          canEditable ? (
            <Button key="3" type="primary" onClick={addAction}>{$t('add.aftersales.address')}</Button>
          ) : (
            <></>
          ),
        ]}
        onReset={() => reload('reset')}
      />
      <AddrForm
        visible={visible}
        supplierList={supplierList}
        value={formValue}
        onClose={onClose}
      ></AddrForm>
    </PageHeaderWrapper>
  );
};

export default connect(({ menu }) => ({
  permissions: menu.permissions,
}))(TableList);
