import React, { useState, useEffect } from 'react';
import { Form } from '@ant-design/compatible';
import { Modal, Button, Upload, notification } from 'antd';
import styles from './index.less';
import { apiImportGoods } from './service';
import ImportListModal from './ImportListModal';
import { ImportConfig } from './config';

const ImportGoods = React.memo(props => {
  const [importFile, setImportFile] = useState([]);
  const [loading, setLoading] = useState(false);
  const [visibleRecord, setVisibleRecord] = useState(false);
  const [config, setConfig] = useState({});

  // 关闭弹窗
  const onCancel = () => {
    props.onHide();
  };
  // 查看导入记录
  const onShowRecord = () => {
    setVisibleRecord(true);
  };
  // 下载模板
  const onDownTemplate = () => {
    if (config.tempPath) {
      window.location.href = config.tempPath;
    } else if (props.onDownload) {
      props.onDownload(config);
    }
  };
  // 导入
  const onImport = async () => {
    if (!importFile.length) {
      notification.error({
        message: $t('select.import.file'),
      });
      return;
    }
    setLoading(true);
    const result = await apiImportGoods(importFile[0], config.type);
    setLoading(false);
    if (result?.success) {
      setImportFile([]);
      notification.success({
        message:
          $t('upload.submitted.check.results.in.history.later'),
      });
    }
  };

  // 导入按钮配置属性
  const uploadFileAttr = {
    name: 'file',
    maxCount: 1,
    fileList: importFile,
    async customRequest(info) {
      setImportFile([info.file]);
      return false;
    },
    accept: '.xlsx',
    showUploadList: true,
    onRemove: () => {
      setImportFile([]);
    },
  };

  useEffect(() => {
    if (props.visible) {
      setConfig(ImportConfig[props.importType]);
    }
  }, [props.visible]);

  return (
    <>
      <Modal
        title={config.title}
        open={props.visible}
        onCancel={onCancel}
        footer={[
          <Button type="link" onClick={onShowRecord} key="btnlook">{$t('view.import.history')}</Button>,
          !config.hideDownload && (
            <Button type="link" onClick={onDownTemplate} key="btndown">{$t('download.template')}</Button>
          ),
          <Button onClick={onCancel} key="btncancel">{$t('close')}</Button>,
          <Button type="primary" loading={loading} onClick={onImport} key="btnimprot">{$t('import')}</Button>,
        ]}
      >
        <Form>
          <Form.Item label={$t('import.file')}>
            <Upload {...uploadFileAttr}>
              <Button type="link" key="btnsel">{$t('choose.file')}</Button>
            </Upload>
            <div className={styles.textDesc}>
              <div>{$t('1.only.supports.by.product')}{config.tip}{$t('import.vertically')}</div>
              <div>{$t('2.supports.excel.files.import.limit')}{config.limitNum}{$t('items')}</div>
            </div>
          </Form.Item>
        </Form>
      </Modal>
      {visibleRecord && (
        <ImportListModal
          visible={visibleRecord}
          type={config.type}
          onCancel={() => setVisibleRecord(false)}
        />
      )}
    </>
  );
});

export default Form.create()(ImportGoods);
