import React, { useRef, useState } from 'react';
import { Form, Input, Button, Checkbox, Card, notification, Popover } from 'antd';
import { history } from 'umi';
import { da } from 'date-fns/locale';
import styles from './style.less';
import { passwordRule, resetPassword } from './service';

const Password = props => {
  const formRef = useRef();
  const [loading, setLoading] = useState(false);
  const goHome = () => {
    formRef.current.resetFields();
    history.push('/');
  };
  const submit = async record => {
    const data = await resetPassword({
      password: record.password,
      newPassword: record.newPassword,
    });
    if (data.code === 2000) {
      notification.success({
        message: $t('edit.successful'),
      });
      setTimeout(() => {
        goHome();
      }, 1000);
      setLoading(false);
    } else {
      setLoading(false);
      notification.error({
        message: data.detail || data.msg,
      });
    }
  };
  const rule = async record => {
    const data = await passwordRule({ password: record.newPassword });
    if (data.code === 2000) {
      submit(record);
    } else {
      setLoading(false);
      notification.warning({
        message: data.detail || data.msg,
      });
    }
  };
  const onFinish = values => {
    setLoading(true);
    rule(values);
  };
  return (
    <div className={styles.div}>
      <Card className={styles.card}>
        <p>{$t('change.password')}</p>

        <Form
          initialValues={{ remember: true }}
          onFinish={onFinish}
          autoComplete="off"
          className={styles.form}
          ref={formRef}
          // labelCol={{
          //   span: 16,
          // }}
          // wrapperCol={{
          //   span: 16,
          // }}
        >
          <Form.Item name="password" rules={[{ required: true, message: $t('enter.current.password') }]}>
            <Input.Password placeholder={$t('enter.current.password')} />
          </Form.Item>
          <Popover
            content={$t('1318.chars.uppercase.lowercase.digits.special.chars')}
            placement="right"
            trigger="hover"
          >
            <Form.Item
              name="newPassword"
              validateFirst
              rules={[
                { required: true, message: $t('enter.new.password') },
                { min: 13, message: $t('min.13char.password') },
                { max: 18, message: $t('max.18char.password') },
                {
                  message: $t('min.13.chars.mixed.casedigitsymbol'),
                  pattern: /(?=.*\d)(?=.*[a-z])(?=.*[A-Z])(?=.*[^a-zA-Z0-9]).{13,18}/,
                },
              ]}
            >
              <Input.Password placeholder={$t('enter.new.password')} />
            </Form.Item>
          </Popover>
          <Form.Item
            name="newPasswordAgain"
            validateFirst
            rules={[
              { required: true, message: $t('enter.new.password') },
              { min: 13, message: $t('min.13char.password') },
              { max: 18, message: $t('max.18char.password') },
              {
                message: $t('min.13.chars.mixed.casedigitsymbol'),
                pattern: /(?=.*\d)(?=.*[a-z])(?=.*[A-Z])(?=.*[^a-zA-Z0-9]).{13,18}/,
              },
              ({ getFieldValue }) => ({
                validator(val, value) {
                  if (value && getFieldValue('newPassword') !== value) {
                    // eslint-disable-next-line prefer-promise-reject-errors
                    return Promise.reject($t('passwords.entered.do.not.match'));
                  }
                  return Promise.resolve();
                },
              }),
            ]}
          >
            <Input.Password placeholder={$t('enter.new.password')} />
          </Form.Item>
          <Form.Item className={styles.btn}>
            <Button type="primary" htmlType="submit" loading={loading}>{$t('confirm.edit')}</Button>
            <Button type="primary" ghost onClick={goHome}>{$t('back.to.home')}</Button>
          </Form.Item>
        </Form>
      </Card>
    </div>
  );
};

export default Password;
