import React, { useState, useRef, useEffect, useCallback } from 'react';
import { Spin, Button, Modal, message, notification } from 'antd';
import { ConsoleSqlOutlined } from '@ant-design/icons';
import { Title, WrapperContainer } from './components/CommonTemplate';
import { TaskTypeSelect } from './components/TaskTypeSelect';
import FormInformationBasic from './components/FormInformationBasic';
import FormPriceOrStock from './components/FormPriceOrStock';
import FormRuleSetting from './components/FormRuleSetting';
import FormRuleVPictures from './components/FormRuleVPictures';
import FormSettlementOthers from './components/FormSettlementOthers';
import {
  getProductDetail,
  merchantCategoryGetAll,
  merchantBrandList,
  merchantSpecList,
  afterSalesAddrsPage,
  merchantgetJdPicList,
  merchantProductAdd,
  // getSupplierList,
  supplierListQuery,
  shopGetBySupplierId,
} from './service';
import { isUrl, filterSendData } from './utils';
import { ServiceContext } from './context';

/**
 * 服务商品改造-商品模块
 * @param {*} router options
 * @returns ReactDOM
 */
const ServiceGoods = options => {
  const { SourceData, categoryList, specListData } = options;

  const basicRef = useRef(null);
  const stockRef = useRef(null);
  const settingRef = useRef(null);
  const picturesRef = useRef(null);
  const settleOtrRef = useRef(null);

  const [pageId, setPageId] = useState(null);
  const [isEdit, setIsEdit] = useState(false); // 是否是编辑状态
  const [productType, setProductType] = useState(4); // 商品状态
  const [pageLoading, setPageLoading] = useState(false); // 页面加载状态
  // const [categoryList, setCategoryList] = useState([]); // 获取三级类目
  const [afterAddressList, setAfterAddressList] = useState([]);
  const [supplierIdList, setSupplierIdList] = useState([]);

  // const [shopList, setShopList] = useState([]);
  const [brandList, setBrandList] = useState([]); // 获取商品牌
  const [specList, setSpecList] = useState([]); // 规格列表
  const [editData, setEditData] = useState({}); // 编辑保存数据
  const baseCheckList = [basicRef, stockRef, settingRef, settleOtrRef, picturesRef]; // 卡卷默认5个，到时改版为实体对应3个
  const [checkFormList, setCheckFormList] = useState(baseCheckList);

  const [specKeyList, setSpecKeyList] = useState([]); // 记录一级规格key字段

  const changeCheckList = proType => {
    const newBaseCheckList =
      proType === 4 ? [...baseCheckList, settingRef, settleOtrRef] : baseCheckList;
    setCheckFormList(newBaseCheckList);
  };

  const resetForm = () => {
    checkFormList.forEach(({ current }) => {
      if (current) {
        current.reset();
      }
    });
  };

  const productChange = task => {
    setProductType(task.type);
    changeCheckList(task.type);
    const timer = setTimeout(() => {
      resetForm();
      clearTimeout(timer);
    }, 1000);
  };

  const handleCancel = () => {
    setPageId(null);
    setIsEdit(false);
    setProductType(4); // 默认写死服务类商品
    setEditData({});
    setSpecKeyList([]);
    resetForm();
    options.onChange(false);
  };

  // 编辑回显详情数据
  const getProductDetailResponse = async id => {
    try {
      const res = await getProductDetail(id);
      if (res && res.data) {
        setProductType(res.data.type);
        setEditData(res.data);
      }
    } catch (err) {
      console.log('接口调用失败！');
    }
  };

  // 获取三级类目分类数据
  // const getMerchantCategory = async () => {
  //   const result = await merchantCategoryGetAll();
  //   // setCategoryList(result.data || []);
  // };
  // 获取商品牌数据
  const getMerchantBrandList = async () => {
    if (!brandList.length) {
      const result = await merchantBrandList();
      setBrandList(result.data || []);
    }
  };
  // 获取规格列表
  const getMerchantSpecList = async () => {
    if (!specList.length) {
      const result = await merchantSpecList();
      setSpecList(result.data || []);
    }
  };

  const getAfterSalesAddrsPage = async () => {
    if (!afterAddressList.length) {
      const result = await afterSalesAddrsPage();
      setAfterAddressList(result.data.records);
    }
  };

  // const getSupplierListResponse = async () => {
  //   if (!shopList.length) {
  //     const result = await supplierListQuery();
  //     console.log('=================>result', result);
  //     setShopList(result.data);
  //   }
  // };

  const sendMerchantProductAdd = async sendData => {
    setPageLoading(true);
    const addResponse = await merchantProductAdd(sendData);
    if (addResponse.data) {
      message.success('添加成功!');
      handleCancel();
    }
    setPageLoading(false);
  };

  const shopGetBySupplierIdResponse = async () => {
    if (!supplierIdList.length) {
      const result = await shopGetBySupplierId();
      setSupplierIdList(result.data);
    }
  };

  const submitEvent = async () => {
    const checkPromiseList = checkFormList.map(({ current }) => current.onCheck());
    const resuslt = await Promise.all(checkPromiseList);
    if (!resuslt.includes(null)) {
      const params = resuslt.reduce((origin, item) => {
        const { temp, ...other } = item;
        origin[temp] = other;
        return origin;
      }, {});

      const sendData = filterSendData(productType, params);
      sendMerchantProductAdd(sendData);
    }
  };

  const getMerchantgetJdPicList = async params => {
    const result = await merchantgetJdPicList(params);
    if (result) {
      const { detailImageList, imageList } = picturesRef.current.getFieldsValue();
      const detailList = result.detailList || [];
      const newImageList = imageList[result.firstSpecValue];
      const carouseList = result.carouseList || [];
      imageList[result.firstSpecValue] = newImageList
        ? [...newImageList, ...carouseList]
        : carouseList;
      picturesRef.current.setFieldsValue({
        // [`imageList[${data.firstSpecValue}]`]: this.state.colorImg[data.firstSpecValue],
        imageList,
        detailImageList: [...detailImageList, ...detailList],
      });
    }
  };

  useEffect(() => {
    (async () => {
      if (!options.visible) {
        return;
      }
      setPageLoading(true);
      await shopGetBySupplierIdResponse();
      // await getSupplierListResponse();
      await getMerchantBrandList();
      await getAfterSalesAddrsPage();
      await getMerchantSpecList();
      if (Object.keys(SourceData).length) {
        // const isService = initData.state && initData.state !== 4;
        setEditData(SourceData);
        setPageId(options.id);
        setProductType(SourceData.productType);
        changeCheckList(SourceData.productType);
        setIsEdit(true);
      }
      setPageLoading(false);
    })();
  }, [SourceData]);

  const onSpecCommonImgEvent = useCallback(
    keys => {
      setSpecKeyList(keys);
    },
    [specKeyList],
  );

  const onEventBus = (event, params) => {
    if (event === 'cloneImg') {
      if (!isUrl(params.skuLink)) {
        notification.open({
          message: '提示',
          description: '请输入正确的URL！',
        });
        return;
      }
      getMerchantgetJdPicList({
        firstSpecId: params.firstSpecId,
        firstSpecValue: params.firstSpecValue,
        secondSpecId: params.secondSpecId,
        secondSpecValue: params.secondSpecValue,
        skuLink: params.skuLink,
        jdSkuInfoUrl: params.skuLink,
      });
    }

    console.log(event, params);
  };

  const providerValue = {
    pageId,
    isEdit,
    productType,
    isCard: productType === 4,
    isService: SourceData.state && SourceData.state !== 4,
    isJDGoods: isEdit && SourceData.pageProductType && +SourceData.pageProductType !== 1,
    onEventBus,
  };

  return (
    <Modal
      visible={options.visible}
      width={1366}
      footer={[
        <Button key="submit" type="primary" loading={pageLoading} onClick={submitEvent}>
          提交
        </Button>,
        <Button key="back" onClick={handleCancel}>
          返回
        </Button>,
      ]}
    >
      <Spin tip="正在加载..." spinning={pageLoading} delay={100}>
        <WrapperContainer>
          <ServiceContext.Provider value={providerValue}>
            <Title title="商品类型" />
            <TaskTypeSelect productType={productType} onChange={productChange} />

            <Title title="商品基本信息编辑" />
            <FormInformationBasic
              ref={basicRef}
              editData={editData.infoMation}
              categoryList={categoryList}
              brandList={brandList}
              afterAddressList={afterAddressList}
              specListData={specListData}
            />

            <Title title="价格与库存" />
            <FormPriceOrStock
              ref={stockRef}
              specList={specList}
              onSpecChange={onSpecCommonImgEvent}
              editData={editData.infoSpecData}
              skuList={editData.skuList}
            />

            <Title title="规则设置" />
            {productType === 4 && (
              <FormRuleSetting
                ref={settingRef}
                editData={editData.serviceItem}
                supplierIdList={supplierIdList}
              />
            )}

            <FormRuleVPictures
              ref={picturesRef}
              specKeyItem={specKeyList}
              editData={editData.infoImageData}
            />
            {productType === 4 && (
              <FormSettlementOthers ref={settleOtrRef} editData={editData.settlementItem} />
            )}
          </ServiceContext.Provider>
        </WrapperContainer>
      </Spin>
    </Modal>
  );
};

export default ServiceGoods;
