import { stringify } from 'querystring';
import _ from 'lodash';
import request from '@/utils/request';
import { APP_CODE } from '@/utils/constants';

const PAGE = 1;
const SIZE = 20;

/**
 * @name 子账号列表
 * @param {Object} params
 * @see http://yapi.quantgroups.com/project/509/interface/api/42832
 */
export function getUserList({ page = PAGE, size = SIZE }) {
  return request.post(`/v2/user/list/${page}/${size}`, {
    data: {},
  });
}

/**
 * @name 子账号详情
 * @param {Object} params
 * @param {number} params.id 子账号id
 * @see http://yapi.quantgroups.com/project/509/interface/api/42856
 */
export function getUserDetail({ id }) {
  return request.get(`/v2/user/info/${id}`);
}

/**
 * @name 新增/修改子账号
 * @see http://yapi.quantgroups.com/project/509/interface/api/42896
 */
export function addUser(params) {
  return request.post('/v2/user/addnew', {
    data: params,
  });
}

/**
 * @name 用户禁用
 * @param {Object} params
 * @param {number} params.id
 * @see http://yapi.quantgroups.com/project/509/interface/api/42864
 */
export function userDisabled({ id }) {
  return request.post(`/v2/user/disabled/${id}`);
}

/**
 * @name 用户启用
 * @param {Object} params
 * @param {number} params.id
 * @see http://yapi.quantgroups.com/project/509/interface/api/42880
 */
export function userEnabled({ id }) {
  return request.post(`/v2/user/enabled/${id}`);
}

/**
 * @name 删除用户
 * @param {Object} params
 * @param {number} params.id
 * @see http://yapi.quantgroups.com/project/509/interface/api/43120
 */
export function userDelete({ id }) {
  return request.post(`/v2/user/delete/${id}`);
}

/**
 * @name 重置/修改密码
 * @param {Object} params
 * @param {string} params.newPassword 新密码
 * @param {string} params.password 确认密码
 * @see http://yapi.quantgroups.com/project/509/interface/api/41800
 */
export function resetPassword(params) {
  return request.post('/v2/user/password/reset', params);
}

/**
 * @name 角色列表-分页
 * @param {Object} params
 * @see http://yapi.quantgroups.com/project/509/interface/api/41816
 */
export function getRoleList({ page, size }) {
  return request.post(`/v2/role/list/${page}/${size}?appCode=${APP_CODE}`, {
    data: {},
  });
}

/**
 * @name 角色详情
 * @param {Object} params
 * @param {number} params.id 角色id
 * @see http://yapi.quantgroups.com/project/509/interface/api/41848
 *
 */
export async function getRoleDetail({ id }) {
  return request.get(`/v2/role/info/${id}`);
}

/**
 * @name 新增角色
 * @param {Object} params
 * @see http://yapi.quantgroups.com/project/509/interface/api/42920
 */
export async function addResource(params) {
  return request.post('/v2/role/addResource', {
    data: params,
  });
}

/**
 * @name 修改角色
 * @param {Object} params
 * @see http://yapi.quantgroups.com/project/509/interface/api/42928
 */
export async function modifyRole(params) {
  return request.post('/v2/role/updateResource', params);
}

/**
 * @name 删除角色
 * @param {Object} params
 * @param {number} params.id 角色id
 * @see http://yapi.quantgroups.com/project/509/interface/api/41832
 */
export async function deleteRole({ id }) {
  return request.post(`/v2/role/delete/${id}`);
}
