import React, { useState, useEffect } from 'react';
import { Form } from '@ant-design/compatible';
import { Modal, Button, Upload, notification } from 'antd';
import styles from './index.less';
import { apiImportGoods } from './service';
import ImportListModal from './ImportListModal';
import { ImportConfig } from './config';

const ImportGoods = React.memo(props => {
  const [importFile, setImportFile] = useState([]);
  const [loading, setLoading] = useState(false);
  const [visibleRecord, setVisibleRecord] = useState(false);
  const [config, setConfig] = useState({});

  // 关闭弹窗
  const onCancel = () => {
    props.onHide();
  };
  // 查看导入记录
  const onShowRecord = () => {
    setVisibleRecord(true);
  };
  // 下载模板
  const onDownTemplate = () => {
    if (config.tempPath) {
      window.location.href = config.tempPath;
    } else if (props.onDownload) {
      props.onDownload(config);
    }
  };
  // 导入
  const onImport = async () => {
    if (!importFile.length) {
      notification.error({
        message: '请选择导入文件',
      });
      return;
    }
    setLoading(true);
    const result = await apiImportGoods(importFile[0], config.type);
    setLoading(false);
    if (result?.success) {
      setImportFile([]);
      notification.success({
        message:
          '已操作上传，稍后上传成功后，请在操作历史中，查看上传结果，如有未上传成功的数据，请根据返回原因，及时处理，修改后重新上传。',
      });
    }
  };

  // 导入按钮配置属性
  const uploadFileAttr = {
    name: 'file',
    maxCount: 1,
    fileList: importFile,
    async customRequest(info) {
      setImportFile([info.file]);
      return false;
    },
    accept: '.xlsx',
    showUploadList: true,
    onRemove: () => {
      setImportFile([]);
    },
  };

  useEffect(() => {
    if (props.visible) {
      setConfig(ImportConfig[props.importType]);
    }
  }, [props.visible]);

  return (
    <>
      <Modal
        title={config.title}
        open={props.visible}
        onCancel={onCancel}
        footer={[
          <Button type="link" onClick={onShowRecord} key="btnlook">
            查看导入记录
          </Button>,
          !config.hideDownload && (
            <Button type="link" onClick={onDownTemplate} key="btndown">
              下载模板
            </Button>
          ),
          <Button onClick={onCancel} key="btncancel">
            关闭
          </Button>,
          <Button type="primary" loading={loading} onClick={onImport} key="btnimprot">
            导入
          </Button>,
        ]}
      >
        <Form>
          <Form.Item label="导入文件">
            <Upload {...uploadFileAttr}>
              <Button type="link" key="btnsel">
                选择文件
              </Button>
            </Upload>
            <div className={styles.textDesc}>
              <div>1、仅支持按商品{config.tip}导入，纵向排列</div>
              <div>2、支持Excel格式文件，导入数量限制{config.limitNum}条</div>
            </div>
          </Form.Item>
        </Form>
      </Modal>
      {visibleRecord && (
        <ImportListModal
          visible={visibleRecord}
          type={config.type}
          onCancel={() => setVisibleRecord(false)}
        />
      )}
    </>
  );
});

export default Form.create()(ImportGoods);
