import { Form, Input, Select, Checkbox, DatePicker } from 'antd';
import React, { useState, useEffect, forwardRef, useImperativeHandle, useContext } from 'react';
import moment from 'moment';
import { WeeksList, formItemLayout } from '../config';
import { ServiceContext } from '../context';

const { Option } = Select;
const { RangePicker } = DatePicker;

const WeekCheckBox = () =>
  WeeksList.map(item => (
    <Checkbox key={item.value} value={item.value} style={{ lineHeight: '32px' }}>
      {item.name}
    </Checkbox>
  ));

const formatTime = (time, crm = 'YYYY-MM-DD HH') => time.format(crm);
const resetTime = (time, crm = 'YYYY-MM-DD HH') => moment(time, crm);

const rangeConfig = {
  rules: [{ type: 'array', required: true, message: 'Please select time!' }],
};

const FormRuleSetting = forwardRef((props, ref) => {
  const { editData } = props;
  const [form] = Form.useForm();
  const customer = useContext(ServiceContext);
  const [imageList, setImageList] = useState([]);
  const [commonImageList, setCommonImageList] = useState([]);
  const [detailImageList, setDetailImageList] = useState([]);
  useEffect(() => {
    if (customer.isEdit) {
      if (!Object.keys(editData).length) return;
      setImageList(editData.commonImageList);
      setCommonImageList(editData.commonImageList);
      setDetailImageList(editData.detailImageList);
      const { serviceItem } = editData.skuList[0];
      form.setFieldsValue({
        useTime: [resetTime(serviceItem.useStartTime), resetTime(serviceItem.useEndTime)],
        purchaseTime: [
          resetTime(serviceItem.purchaseStartTime),
          resetTime(serviceItem.purchaseEndTime),
        ], // 购买开始时间
        shopIds: serviceItem.shopIds || [], // 适用门店列表
        unavailableDate: serviceItem.unavailableDate, // 不可用日期
        useTimeDescription: serviceItem.useTimeDescription, // 使用时间
        useMethod: serviceItem.useMethod, // 使用方法
        ruleDescription: serviceItem.ruleDescription, // 规则说明
        applyScope: serviceItem.applyScope, // 适用范围
        tips: serviceItem.tips, // 温馨提示
      });
    }
  }, [customer.isEdit, editData]);

  const onCheck = async () => {
    try {
      const { useTime, purchaseTime, ...values } = await form.validateFields();
      return {
        useStartTime: formatTime(useTime[0]),
        useEndTime: formatTime(useTime[1]),

        purchaseStartTime: formatTime(purchaseTime[0]),
        purchaseEndTime: formatTime(purchaseTime[1]),
        ...values,
      };
    } catch (errorInfo) {
      return null;
    }
  };

  useImperativeHandle(ref, () => ({
    onCheck,
    reset: form.resetFields,
  }));

  const onFinish = values => {
    console.log('Received values of form: ', values);
  };
  return (
    <>
      <Form
        {...formItemLayout}
        form={form}
        name="register"
        onFinish={onFinish}
        initialValues={{
          useTime: [], // 使用开始时间
          // useEndTime: '', // 使用结束时间
          purchaseTime: [], // 购买开始时间
          // purchaseEndTime: '2022-07-27 06', // 购买结束时间
          shopIds: [], // 适用门店列表
          unavailableDate: [], // 不可用日期
          useTimeDescription: '', // 使用时间
          useMethod: '', // 使用方法
          ruleDescription: '', // 规则说明
          applyScope: '', // 适用范围
          tips: '', // 温馨提示

          imageList,
          commonImageList,
          detailImageList,
        }}
        scrollToFirstError
      >
        <Form.Item name="useTime" label="购买时间" {...rangeConfig}>
          <RangePicker showTime format="YYYY-MM-DD HH" />
        </Form.Item>
        <Form.Item name="purchaseTime" label="有效期" {...rangeConfig}>
          <RangePicker showTime format="YYYY-MM-DD HH" />
        </Form.Item>
        <Form.Item
          name="shopIds"
          label="适用门店"
          rules={[{ required: true, message: '请选择适用门店!', type: 'array' }]}
        >
          <Select mode="multiple" placeholder="请选择适用门店">
            <Option value={1}>Male</Option>
            <Option value={2}>Female</Option>
            <Option value={3}>Other</Option>
          </Select>
        </Form.Item>

        <Form.Item
          name="unavailableDate"
          label="不可用日期"
          rules={[{ required: true, message: '请选择不可用日期!', type: 'array' }]}
        >
          <Checkbox.Group>{WeekCheckBox()}</Checkbox.Group>
        </Form.Item>
        <Form.Item
          name="useTimeDescription"
          label="使用时间"
          rules={[{ required: false, message: '请输入描述！' }]}
        >
          <Input.TextArea
            showCount
            maxLength={100}
            placeholder="例如：11:00-12:00;14:00-17:00可用，其他时间不可用"
          />
        </Form.Item>
        <Form.Item
          name="useMethod"
          label="使用方法"
          rules={[{ required: false, message: '请输使用方法，200字以内！' }]}
        >
          <Input.TextArea showCount maxLength={200} placeholder="请输使用方法，200字以内！" />
        </Form.Item>
        <Form.Item
          name="ruleDescription"
          label="规则说明"
          rules={[{ required: false, message: '请输规则说明，200字以内！' }]}
        >
          <Input.TextArea showCount maxLength={200} placeholder="请输规则说明，200字以内！" />
        </Form.Item>
        <Form.Item
          name="applyScope"
          label="适用范围"
          rules={[{ required: false, message: '请输入适用范围' }]}
        >
          <Input.TextArea
            showCount
            maxLength={50}
            placeholder="请输入适用范围，50字以内 例如：全场通用"
          />
        </Form.Item>
        <Form.Item
          name="tips"
          label="温馨提示"
          rules={[{ required: false, message: '请输入温馨提示' }]}
        >
          <Input.TextArea
            showCount
            maxLength={200}
            placeholder="请输入温馨提示，200字以内 例如：全场通用例如：\n不兑零"
          />
        </Form.Item>
      </Form>
    </>
  );
});

export default FormRuleSetting;
