import React, { useState, useEffect, useRef } from 'react';
import { useHistory } from 'react-router-dom';
import { ProTable } from '@ant-design/pro-components';
import { Button, notification } from 'antd';
import { PlusOutlined } from '@ant-design/icons';
import { takeawayGoodsColumn } from './staticData/goods';
import utilStyle from '@/utils/utils.less';
import { apiTakeawayList, apiSelPickSelfList, apiMealInfoDel, apiMealInfoUpdate } from './service';
import { getEnterpriseList, getShopList } from './service/bll';
import SaleDateModal from './components/SaleDateModal';
import SaleSectionModal from './components/SaleSectionModal';
import GoodPriceModal from './components/GoodPriceModal';
import GoodSortModal from './components/GoodSortModal';
import { getToUrlQuery } from '@/utils/utils';

// 企业外卖商品
const TakeawayGoods = () => {
  const history = useHistory();
  const refTable = useRef();
  const [pageLoaded, setPageLoaded] = useState(false); // 可售日期弹窗
  const [visibleSaleDate, setVisibleSaleDate] = useState(false); // 可售日期弹窗
  const [visibleSaleSection, setVisibleSaleSection] = useState(false); // 可售餐段弹窗
  const [visiblePrice, setVisiblePrice] = useState(false); // 修改企业商品价格弹窗
  const [visibleSort, setVisibleSort] = useState(false); // 商品排序弹窗
  const [enterprises, setEnterprises] = useState({}); // 企业列表
  const [shopEnum, setShopEnum] = useState({}); // 店铺列表
  const [activeKey, setActiveKey] = useState('');
  const [enterpriseId, setEnterpriseId] = useState();
  const [pickSelfList, setPickSelfList] = useState([]); // 取餐点列表
  const [recordID, setRecordID] = useState(''); // 编辑的记录ID

  // 刷新列表
  const onRefresh = () => {
    if (pageLoaded) {
      refTable.current.reloadAndRest();
      // refTable.current.reload();
    }
  };

  // 搜索商品列表
  const searchList = async params => {
    const data = {
      page: params.current || 1,
      size: params.pageSize || 10,
      data: Object.assign({}, params, {
        enterpriseId,
        pickSelfId: activeKey,
      }),
    };
    const res = await apiTakeawayList(data);
    if (res && res.records) {
      return {
        data: res.records || [],
        total: res.total || 0,
      };
    }
    return {
      data: [],
      total: 0,
    };
  };
  // 删除商品
  const onDel = async id => {
    await apiMealInfoDel({
      id,
    });
    refTable.current.reload();
    notification.success({
      message: '删除成功',
    });
  };
  // 跳转添加商品
  const toAdd = async () => {
    const params = getToUrlQuery();
    const query = Object.assign(
      {
        id: enterpriseId,
        name: enterprises[enterpriseId].text,
      },
      params,
    );
    history.push({
      pathname: '/takeawayGoodsInfo',
      query,
    });
  };
  // 修改商品是否列出
  const onChangeFlag = async (id, checked) => {
    const params = {
      id,
      showFlag: checked ? 1 : 0,
    };
    await apiMealInfoUpdate(params);
    notification.success({ message: '保存成功' });
    onRefresh();
  };

  // 根据企业ID获取取餐点
  const getPickSelf = async id => {
    const res = await apiSelPickSelfList({ enterpriseId: id });
    if (res && res.data && res.data.length) {
      setPickSelfList(
        res.data.map(item => ({
          key: `${item.pickSelfId}`,
          label: <span>{item.pickSelfName}</span>,
        })),
      );
      const pid = res.data[0].pickSelfId;
      setActiveKey(`${pid}`);
      if (pageLoaded) {
        onRefresh();
      }
    } else {
      setPickSelfList([]);
      setActiveKey('');
      if (pageLoaded) {
        onRefresh();
      }
    }
  };

  // 改变企业
  const onChangeEnterprise = v => {
    setEnterpriseId(v);
    getPickSelf(v);
  };

  // 获取企业列表
  const getList = async () => {
    const obj = await getEnterpriseList();
    if (obj.list && Object.keys(obj.list).length) {
      setEnterprises(obj.list);
      setEnterpriseId(`${obj.id}`);
      await getPickSelf(obj.id);
      setPageLoaded(true);
    }
  };

  useEffect(() => {
    getList();
  }, []);

  const options = {
    setVisibleSaleDate,
    setVisibleSaleSection,
    setVisiblePrice,
    setVisibleSort,
    onDel,
    setRecordID,
    enterprises,
    onChangeEnterprise,
    onChangeFlag,
    onRefresh,
    shopEnum,
  };

  return (
    <div className={utilStyle.formPageBox}>
      {pageLoaded && (
        <ProTable
          search={{
            span: 6,
            className: utilStyle.formSearch,
            collapsed: false,
            collapseRender: () => null,
          }}
          actionRef={refTable}
          tableClassName={utilStyle.formTable}
          columns={takeawayGoodsColumn(options)}
          request={params => searchList({ ...params })}
          rowKey={r => r.id}
          expandIconColumnIndex={10}
          bordered
          options={false}
          form={{ initialValues: { enterpriseId } }}
          toolbar={{
            menu: {
              type: 'tab',
              activeKey,
              items: pickSelfList,
              onChange: key => {
                setActiveKey(`${key}`);
                onRefresh();
              },
            },
            actions: [
              <Button key="1" icon={<PlusOutlined />} type="primary" onClick={toAdd}>
                添加外卖商品
              </Button>,
            ],
          }}
          scroll={{ x: '100%', y: 400 }}
        />
      )}
      {/* 可售日期弹窗 */}
      {visibleSaleDate && (
        <SaleDateModal
          visible={visibleSaleDate}
          id={recordID}
          handleRefresh={() => onRefresh()}
          handleClose={() => setVisibleSaleDate(false)}
        />
      )}
      {/* 可售餐段弹窗 */}
      {visibleSaleSection && (
        <SaleSectionModal
          visible={visibleSaleSection}
          id={recordID}
          handleRefresh={() => onRefresh()}
          handleClose={() => setVisibleSaleSection(false)}
        />
      )}
      {/* 修改企业商品价格弹窗 */}
      {visiblePrice && (
        <GoodPriceModal
          visible={visiblePrice}
          id={recordID}
          handleRefresh={() => onRefresh()}
          handleClose={() => setVisiblePrice(false)}
        />
      )}
      {/* 商品排序弹窗 */}
      {visibleSort && (
        <GoodSortModal
          visible={visibleSort}
          id={recordID}
          handleRefresh={() => onRefresh()}
          handleClose={() => setVisibleSort(false)}
        />
      )}
    </div>
  );
};

export default TakeawayGoods;
