import EventEmitter from 'events';
import { Modal } from 'antd';

class Socket extends EventEmitter {
  event = new EventEmitter();

  constructor(options) {
    super();
    this.options = options;
    this.reconnectCount = 0;
    this.socket = null;
    this.taskRemindInterval = null;
    this.connected = false;
    this.waitingSendData = [];
    this.reconnectCount = 0;
    return this;
  }

  connection = () => {
    const { url, timeout = 0 } = this.options;
    // 检测当前浏览器是什么浏览器来决定用什么socket
    if ('WebSocket' in window) {
      console.log('WebSocket');
      this.socket = new WebSocket(url);
    } else if ('MozWebSocket' in window) {
      console.log('MozWebSocket');
      // eslint-disable-next-line no-undef
      this.socket = new MozWebSocket(url);
    } else {
      console.log('SockJS');
      // eslint-disable-next-line no-undef
      this.socket = new SockJS(url);
    }

    // 链接回调
    this.socket.onopen = this.onopen;
    this.socket.onmessage = this.onmessage;
    this.socket.onclose = this.onclose;
    this.socket.onerror = this.onerror;
    this.socket.sendMessage = this.sendMessage;

    // 检测返回的状态码 如果socket.readyState不等于1则连接失败，关闭连接
    if (timeout) {
      const time = setTimeout(() => {
        if (this.socket && this.socket.readyState !== 1) {
          console.log('主动触发关闭');
          this.socket.close();
        }
        clearInterval(time);
      }, timeout);
    }
  };

  // 连接成功触发
  onopen = () => {
    console.log('ws:连接成功');
    this.connected = true;
    this.checkWaitingData();
    this.event.emit('open');
  };

  // 后端向前端推得数据
  onmessage = msg => {
    console.log('接收数据:', msg);
    this.event.emit('message', msg);
    // 打印出后端推得数据
  };

  // 关闭连接触发
  onclose = e => {
    this.connected = false; // 关闭将标识符改为true
    console.log('关闭socket收到的数据');
    this.event.emit('close', e);
    // 根据后端返回的状态码做操作
    // 我的项目是当前页面打开两个或者以上，就把当前以打开的socket关闭
    // 否则就20秒重连一次，直到重连成功为止
    if (this.reconnectCount > 10) {
      this.reconnectCount = 0;
      return;
    }
    const reconnect = () => {
      clearTimeout(this.taskRemindInterval);
      this.taskRemindInterval = setTimeout(() => {
        if (!this.connected) {
          this.reconnectCount++;
          this.connection();
          reconnect();
        }
      }, 2000);
    };
    reconnect();
  };

  onerror = e => {
    this.socket = null;
    this.event.emit('error', e);
  };

  sendMessage = value => {
    // 向后端发送数据
    if (this.socket) {
      if (!this.connected) {
        this.waitingSendData.unshift(value);
        return;
      }
      this.socket.send(JSON.stringify(value));
    }
  };

  checkWaitingData() {
    if (this.waitingSendData.length) {
      this.sendMessage(this.waitingSendData.splice(0, 1));
      this.checkWaitingData();
    }
  }
}

export default Socket;
