import React, { useState, useEffect, useContext, forwardRef, useImperativeHandle } from 'react';
import { Form, Button } from 'antd';
import { ServiceContext } from '../context';
import { TaskList, formItemLayout } from '../config';
import UploadImage from './UploadImage';
import commonStyle from '../common.less';

const { imgConfig: defaultImgConfig } = TaskList[0];

const FormRuleVPictures = forwardRef((props, ref) => {
  const { editData, specKeyItem } = props;
  const [form] = Form.useForm();
  const [imageList, setImageList] = useState({});
  const [commonImageList, setCommonImageList] = useState([]);
  const [detailImageList, setDetailImageList] = useState([]);

  const customer = useContext(ServiceContext);
  useEffect(() => {
    if (customer.isEdit) {
      if (editData) {
        setImageList(editData.imageList);
        setCommonImageList(editData.commonImageList); // 编辑状态下设置公共图
        setDetailImageList(editData.detailImageList); // 编辑状态下设置详情图
        form.setFieldsValue({
          imageList: editData.imageList,
          commonImageList: editData.commonImageList,
          detailImageList: editData.detailImageList,
        });
      }
    }
  }, [customer.isEdit, editData]);

  useEffect(() => {
    if (customer.isCard) return;
    if (specKeyItem.length) {
      const newImageList = specKeyItem.reduce((origin, item) => {
        const showItem = imageList[item] || [];
        origin[item] = showItem;
        return origin;
      }, {});
      setImageList(newImageList);
    }
  }, [specKeyItem]);

  const onCheck = async () => {
    try {
      const values = await form.validateFields();
      values.temp = 'infoImageData';
      return values;
    } catch (errorInfo) {
      return null;
    }
  };

  useImperativeHandle(ref, () => ({
    onCheck,
    setFieldsValue: form.setFieldsValue,
    getFieldsValue: form.getFieldsValue,
    reset: () => {
      form.resetFields();
      setImageList({});
      setCommonImageList([]);
      setDetailImageList([]);
    },
  }));

  const onFinish = values => {
    console.log('Received values of form: ', values);
  };
  const onPictureSuccessEvent = (imgList, key) => {
    const newImgList = { ...imageList, [key]: imgList };
    setImageList(newImgList);
    form.setFieldsValue({
      imageList: newImgList,
    });
  };

  const onCommonSuccessEvent = imgList => {
    setCommonImageList(imgList);
    form.setFieldsValue({
      commonImageList: imgList,
    });
  };

  const onDetailSuccessImageList = imgList => {
    setDetailImageList(imgList);
    form.setFieldsValue({
      detailImageList: imgList,
    });
  };

  const [{ imgConfig }] = TaskList.filter(item => item.type === customer.productType);

  return (
    <>
      <Form
        {...formItemLayout}
        form={form}
        name="register"
        onFinish={onFinish}
        initialValues={{
          imageList: {},
          commonImageList: [],
          detailImageList: [],
        }}
        scrollToFirstError
      >
        <Form.Item
          label="封面图片"
          name="commonImageList"
          extra={`建议尺寸: ##宽##高 (${commonImageList.length} / 1) `}
          rules={[
            {
              required: imgConfig.commonImageList.rule,
              type: 'array',
              message: '请上传图片！',
              validateTrigger: 'submit',
            },
            {
              validator(rule, value, callback) {
                if (customer.productType !== 1) callback();
                const checkImageList = form.getFieldValue('imageList');
                const check = Object.keys(checkImageList).length;
                return check > 0 ? callback() : callback('请上传封面图片');
              },
              validateTrigger: 'submit',
            },
          ]}
        >
          <UploadImage
            disabled={customer.isService}
            name="commonImageList"
            limit={1}
            pictures={commonImageList}
            setPictureList={list => onCommonSuccessEvent(list)}
          />
        </Form.Item>

        {!customer.isCard &&
          Object.keys(imageList).map(key => (
            <Form.Item
              key={key}
              label={`商品图片(${key})`}
              name={['imageList', key]}
              extra={`建议尺寸: ##宽##高 (${imageList[key].length} / 11) `}
              rules={[
                {
                  required: imgConfig.imageList.rule,
                  type: 'array',
                  message: '请上传图片！',
                  validateTrigger: 'submit',
                },
              ]}
            >
              <UploadImage
                disabled={customer.isService}
                name={key}
                limit={11}
                pictures={imageList[key]}
                setPictureList={list => onPictureSuccessEvent(list, key)}
              />
              <Button className={commonStyle.pullImage} type="primary">
                拉取公共图
              </Button>
            </Form.Item>
          ))}

        <Form.Item
          label="商品详情图"
          name="detailImageList"
          extra={`最多上传30张，${detailImageList.length} / 30`}
          rules={[
            {
              type: 'array',
              required: imgConfig.detailImageList.rule,
              message: '请上传商品详情图！',
              validateTrigger: 'submit',
            },
          ]}
        >
          <UploadImage
            disabled={customer.isService}
            name="detailImageList"
            pictures={detailImageList}
            setPictureList={list => onDetailSuccessImageList(list)}
          />
        </Form.Item>
      </Form>
    </>
  );
});

export default FormRuleVPictures;
