// import fileSaver from 'file-saver';
import request from '@/utils/request';
import config from '../../../config/env.config';
import { stringify } from 'qs';
import _ from 'lodash';

const { goodsApi, kdspApi } = config;
const headers = {
  'Content-Type': 'application/x-www-form-urlencoded',
};

/**
 * 商品列表
 * yapi: http://yapi.quantgroups.com/project/389/interface/api/23814
 */
export async function searchList(params) {
  return request.post('/product/api/merchant/page', {
    prefix: goodsApi,
    data: stringify(params),
    headers,
    role: true,
  });
}

// 新增商品
export async function addGoods(params) {
  return request.post('/product/api/merchant/add', {
    prefix: goodsApi,
    data: params,
  });
}

// 编辑商品

export async function editGoods(params) {
  return request.post('/product/api/merchant/edit', {
    prefix: goodsApi,
    data: params,
  });
}

// 获取商品品牌
export async function getBrandList() {
  return request.post('/product/brand/api/merchant/list', {
    prefix: goodsApi,
  });
}

// 获取类目关联属性
export async function apiGetAttribute(categoryId) {
  const data = await request.get(
    `/api/kdsp/category/template/ref/attribute/detail?categoryId=${categoryId}`,
    {
      prefix: goodsApi,
    },
  );
  return data;
}
// 编辑--获取详情
export async function spuDetail(params) {
  return request.post('/product/api/merchant/detail', {
    prefix: goodsApi,
    params,
    headers,
  });
}

// 商品规格
export async function getSpecList() {
  return request.post('/product/spec/api/merchant/list', {
    prefix: goodsApi,
  });
}

// 查询图片素材
export async function getImageInfo(spuNo) {
  return request.post('/product/api/merchant/imageInfo', {
    params: { spuNo },
    prefix: goodsApi,
    headers,
  });
}

// 商品分类
export async function categoryList() {
  return request.post('/product/category/api/merchant/getAll', {
    prefix: goodsApi,
  });
}
/**
 * 商品分类
 * type 商品类型：1-实物类，2-虚拟类，4-服务类
 * */
export async function apiCategoryListType(type) {
  return request.get(`/product/category/getByProductType/${type}`, {
    prefix: goodsApi,
  });
}

// 批量修改
export async function uploadFile(file) {
  const params = new FormData();
  params.append('file', file);
  const data = await request.post('/product/api/merchant/BatchUpdateStock', {
    prefix: goodsApi,
    data: params,
    notTip: true,
  });
  return data;
}

// 商品修改日志
export async function changeLog(params) {
  return request.post('/product/logs/api/merchant/page', {
    data: stringify(params),
    prefix: goodsApi,
    headers,
  });
}

// 审核详情日志
export async function productMerchantLog(productId) {
  return request.get('/product/api/merchant/auditLog', {
    params: {
      productId,
    },
    prefix: goodsApi,
  });
}

// 拉去京东图片
export async function getJdPicList(params) {
  const { data } = await request.post('/product/api/merchant/item/getJdPicList', {
    data: stringify(params),
    prefix: goodsApi,
    headers,
  });
  return data;
}

// 图片上传
export async function uploadImg(files) {
  const params = new FormData();
  files.forEach(file => params.append('file', file));
  const data = await request.post('/image/api/merchant/upload', {
    prefix: goodsApi,
    data: params,
  });
  return data;
}

// 修改商品库存
export async function updateStock(params) {
  const data = await request.post('/product/item/api/merchant/updateStock', {
    prefix: goodsApi,
    data: stringify(params),
    headers,
  });
  if (data.businessCode === '0000') {
    return null;
  }
  return data.msg;
}

// 获取虚拟商品类目
export async function getVirtualCategory() {
  const data = await request.post('/product/category/api/merchant/getByParentId', {
    prefix: goodsApi,
    data: stringify({ id: 100018 }),
    headers,
  });
  return data;
}

// 查询商品属性列表
export async function specList() {
  return request.get('/product/api/merchant/spec', {
    prefix: goodsApi,
  });
}
// sku绑定模板
export async function getTemplate(params) {
  const data = await request.post('/api/kdsp/sku/template', {
    data: params,
    prefix: goodsApi,
  });
  return data;
}

// 不配送区域模板列表
export async function getTemplateList(params) {
  const data = await request.post('/api/kdsp/template/list', {
    data: stringify(params),
    prefix: goodsApi,
    headers,
  });
  return data;
}
// 查询供应商售后地址
export async function queryAllAfterAddress() {
  const params = {
    pageSize: 100,
    pageNo: 1,
  };
  const data = request.post('/api/kdsp/supplier/after-sales-addrs-page', {
    prefix: kdspApi,
    data: stringify(_.omitBy(params, v => !v)),
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded',
    },
  });
  return data;
}

// 商品售后地址设置
export async function getAfterAddress(params) {
  const data = await request.post('/api/kdsp/updateSkuAfterAddress', {
    data: params,
    prefix: goodsApi,
  });
  return data;
}

/**
 * 商品是否可以做库存变更
 * skuIds: 多个用英文逗号隔开
 */
export const apiEnableUpdataStock = skuIds =>
  request.get(`/api/kdsp/sku/can/stockChange?skuIds=${skuIds}`, {
    prefix: goodsApi,
  });

// 商品明细导出
export const apiGoodsInfosExport = params =>
  request.post('/api/kdsp/sku/export', {
    prefix: goodsApi,
    data: params,
    responseType: 'arrayBuffer',
    headers: {
      'Content-Type': 'application/json;charset=UTF-8',
    },
  });

// 新建商品审核记录
export const apiCreateGoodsLog = params =>
  request.post('/api/kdsp/product/audit/create', {
    data: params,
    prefix: goodsApi,
  });

// 查询sku最后一条库存变更记录
export const apiQueryLastChangeLog = skuId =>
  request.get(`/api/kdsp/sku/last/stockChange/record?skuId=${skuId}`, {
    prefix: goodsApi,
  });

// 商品上下架
export const apiChangeStateGoods = async params => {
  const data = await request.post('/product/api/merchant/pushedOrOffline/batch', {
    data: stringify(params),
    prefix: goodsApi,
    headers,
  });
  return data;
};

// 查询sku最后一条审核记录
export const apiQueryLastAuditRecord = skuId =>
  request.get(`/api/kdsp/sku/last/audit/record?skuId=${skuId}`, {
    prefix: goodsApi,
  });

// 商品草稿详情
export const apiDraftDetail = draftId =>
  request.get(`/api/merchants/drafts/detail?id=${draftId}`, {
    prefix: goodsApi,
  });

// 删除商品草稿
export const apiDeleteDraft = draftId =>
  request.get(`/api/merchants/drafts/delete?id=${draftId}`, {
    prefix: goodsApi,
  });

// 商品草稿列表
export async function apiDraftList(data) {
  return request.post('/api/merchants/drafts/list', {
    prefix: goodsApi,
    data,
  });
}
