import React, { useState, useRef, useEffect } from 'react';
import { PageHeaderWrapper } from '@ant-design/pro-layout';
import { Spin, Button } from 'antd';
import { Title, WrapperContainer } from './components/CommonTemplate';
import { TaskTypeSelect } from './components/TaskTypeSelect';
import { Task } from './type';
import FormInformationBasic from './components/FormInformationBasic';
import FormPriceOrStock from './components/FormPriceOrStock';
import FormRuleSetting from './components/FormRuleSetting';
import FormSettlementOthers from './components/FormSettlementOthers';
import commonStyle from './common.less';
import {
  getProductDetail,
  merchantCategoryGetAll,
  merchantBrandList,
  merchantSpecList,
} from './service';
import { ServiceContext } from './context';

/**
 * 服务商品改造-商品模块
 * @param {*} router options
 * @returns ReactDOM
 */
const ServiceGoods = options => {
  const pageId = +options.match.params.id; // pageId: 0-新增，1-修改，2-查看

  const basicRef = useRef(null);
  const stockRef = useRef(null);
  const settingRef = useRef(null);
  const settleOtrRef = useRef(null);

  const [isEdit, setIsEdit] = useState(false); // 是否是编辑状态
  const [productType, setProductType] = useState(1); // 商品状态
  const [pageLoading, setPageLoading] = useState(false); // 页面加载状态
  const [categoryList, setCategoryList] = useState([]); // 获取三级类目
  const [brandList, setBrandList] = useState([]); // 获取商品牌
  const [specList, setSpecList] = useState([]); // 规格列表
  const [editData, setEditData] = useState({}); // 编辑保存数据

  const productChange = (task: Task): void => {
    setProductType(task.type);
  };

  const submitEvent = async () => {
    const resuslt = await Promise.all([
      basicRef.current.onCheck(),
      stockRef.current.onCheck(),
      settingRef.current.onCheck(),
      settleOtrRef.current.onCheck(),
    ]);
    console.log(resuslt);
  };

  // 编辑回显详情数据
  const getProductDetailResponse = async () => {
    try {
      const res = await getProductDetail(pageId);
      if (res && res.data) {
        setProductType(res.data.type);
        setEditData(res.data);
      }
    } catch (err) {
      console.log('接口调用失败！');
    }
  };

  // 获取三级类目分类数据
  const getMerchantCategory = async () => {
    const result = await merchantCategoryGetAll();
    setCategoryList(result.data || []);
  };
  // 获取商品牌数据
  const getMerchantBrandList = async () => {
    const result = await merchantBrandList();
    setBrandList(result.data || []);
  };
  // 获取规格列表
  const getMerchantSpecList = async () => {
    const result = await merchantSpecList();
    setSpecList(result.data || []);
  };

  useEffect(() => {
    (async () => {
      setPageLoading(true);
      // await getMerchantCategory();
      // await getMerchantBrandList();
      await getMerchantSpecList();
      if (pageId !== 0) {
        setIsEdit(true);
        await getProductDetailResponse();
      }
      setPageLoading(false);
    })();
  }, [pageId]);

  return (
    <Spin tip="正在加载..." spinning={pageLoading} delay={100}>
      <PageHeaderWrapper className={commonStyle.header}>
        <WrapperContainer>
          <ServiceContext.Provider value={{ pageId, isEdit, productType }}>
            <Title title="商品类型" />
            <TaskTypeSelect productType={productType} onChange={productChange} />

            <Title title="商品基本信息编辑" />
            <FormInformationBasic
              ref={basicRef}
              editData={editData}
              categoryList={categoryList}
              brandList={brandList}
            />

            <Title title="价格与库存" />
            <FormPriceOrStock ref={stockRef} specList={specList} editData={editData} />

            <Title title="规则设置" />
            <FormRuleSetting ref={settingRef} editData={editData} />

            <FormSettlementOthers ref={settleOtrRef} editData={editData} />
            <Button type="primary" onClick={submitEvent}>
              Register
            </Button>
          </ServiceContext.Provider>
        </WrapperContainer>
      </PageHeaderWrapper>
    </Spin>
  );
};

export default ServiceGoods;
