import React from 'react';
import { Input, InputNumber, Select, Checkbox, Radio } from 'antd';
import { jsonToArray } from '@/utils/utils';
import moment from 'moment';
import MealSection from '../MealSection';
import { mealType, boolOptions, hideOptions } from '../../staticData/index';
import MealFormListLimit from './MealFormListLimit';

const CheckboxGroup = Checkbox.Group;
const RadioGroup = Radio.Group;

const hideEnums = { hidePrice: '隐藏商品价格', hideImage: '隐藏商品图片' };
const weekEnums = { 1: '是', 0: '否' };

// 校验时间
const checkTime = (arr, current) => {
  let valid = false;
  arr.forEach(item => {
    if (current < item.endTime) {
      valid = true;
    }
  });
  return valid;
};
// 验证餐段
const validateMeals = (form, meals) => {
  const { mealTimePeriod = [] } = form.getFieldValue();
  const arr = [];
  let validTime = false;
  mealTimePeriod.forEach(item => {
    if (item && meals[item.mealPeriodType]) {
      const obj = { ...item };
      if (obj.time?.length === 2) {
        obj.beginTime = moment(obj.time[0]).format('HH:mm');
        obj.endTime = moment(obj.time[1]).format('HH:mm');
        if (checkTime(arr, obj.beginTime, meals[item.mealPeriodType])) {
          validTime = true;
        }
        arr.push(obj);
      }
    }
  });
  if (validTime) {
    // eslint-disable-next-line prefer-promise-reject-errors
    return Promise.reject('时间段不能交叉！');
  }
  return Promise.resolve();
};

export const getFormItem = options => {
  const { id, pickSelfList, onChangeMealType, onChangeMealSection, meals, form } = options;
  return [
    {
      type: 'Form.Item',
      label: '企业名称',
      name: 'name',
      wrapperCol: { span: 20 },
      rules: [{ required: true, message: '请输入企业名称' }],
      component: <Input placeholder="请输入企业名称" />,
    },
    {
      type: 'Form.Item',
      disabled: id,
      label: '企业取餐点',
      wrapperCol: { span: 20 },
      name: 'pickSelfIds',
      rules: [{ required: true, message: '请选择企业取餐点' }],
      component: (
        <Select
          options={pickSelfList}
          mode="multiple"
          showSearch
          filterOption={(v, option) =>
            (option?.label ?? '').toLowerCase().includes(v.toLowerCase())
          }
        />
      ),
    },
    {
      type: 'Form.Item',
      label: '企业截止时间',
      wrapperCol: { span: 20 },
      name: 'endOrderTime',
      rules: [{ required: true, message: '请输入企业截止时间' }],
      component: <InputNumber min={0} max={600} addonAfter="分钟" />,
      extra: <span>企业员工下单的截至时间，仅支持正整数，单位为分钟。</span>,
    },
    {
      type: 'Form.Item',
      label: '餐品类型',
      wrapperCol: { span: 20 },
      name: 'mealType',
      rules: [{ required: true, message: '请选择餐品类型' }],
      component: <CheckboxGroup options={jsonToArray(mealType)} onChange={onChangeMealType} />,
    },
    {
      type: 'Form.Item',
      label: '餐段配置',
      wrapperCol: { span: 12 },
      name: 'mealTimePeriod',
      rules: [{ required: true, message: '请选择餐段配置' }],
      component: (
        <MealSection
          meals={meals}
          validateMeals={() => validateMeals(form, meals)}
          onChange={onChangeMealSection}
        />
      ),
    },
    {
      name: 'Divider',
      type: 'Divider',
      component: '企业单笔消费限额',
    },
    {
      name: 'Form.List',
      type: 'Form.List',
      component: '',
    },
    {
      type: 'Form.Item',
      label: '商品隐藏信息',
      wrapperCol: { span: 20 },
      name: 'hideInfo',
      component: <CheckboxGroup options={hideOptions} />,
    },
    {
      type: 'Form.Item',
      label: '是否周预览',
      wrapperCol: { span: 20 },
      name: 'weekPreview',
      rules: [{ required: true, message: '请选择是否周预览' }],
      component: <RadioGroup options={boolOptions} />,
    },
  ];
};

// 获取 schemaForm 的 columns
export const getBaseFormItem = options => {
  const {
    id,
    pickSelfList,
    selectedMealTypes,
    onChangeMealType,
    onChangeMealSection,
    meals,
    form,
  } = options;

  const baseColumn = [
    {
      title: '企业名称',
      dataIndex: 'name',
      formItemProps: {
        rules: [{ required: true, message: '请输入企业名称' }],
      },
    },
    {
      hideInForm: !!id,
      title: '企业取餐点',
      dataIndex: 'pickSelfIds',
      valueType: 'select',
      valueEnum: pickSelfList,
      fieldProps: {
        mode: 'multiple',
      },
      formItemProps: {
        rules: [{ required: true, message: '请选择企业取餐点' }],
      },
    },
    {
      title: '企业截止时间',
      dataIndex: 'endOrderTime',
      valueType: 'digit',
      formItemProps: {
        rules: [{ required: true, message: '请输入企业截止时间' }],
        addonAfter: '分钟',
        extra: <span>企业员工下单的截至时间，仅支持正整数，单位为分钟。</span>,
      },
    },
    {
      title: '餐品类型',
      dataIndex: 'mealType',
      valueType: 'checkbox',
      valueEnum: mealType,
      formItemProps: {
        rules: [{ required: true, message: '请选择餐品类型' }],
      },
      fieldProps: {
        onChange: onChangeMealType,
        name: 'mealType',
      },
    },
    {
      title: '餐段配置',
      dataIndex: 'mealTimePeriod',
      formItemProps: {
        rules: [{ required: true, message: '请选择餐段配置' }],
      },
      fieldProps: {
        copyIconProps: false, // 隐藏复制行按钮
        deleteIconProps: false, // 隐藏删除行按钮
        creatorButtonProps: false, // 隐藏添加行按钮
      },
      renderFormItem: () => (
        <MealSection
          meals={meals}
          validateMeals={() => validateMeals(form, meals)}
          onChangeSection={onChangeMealSection}
        />
      ),
    },
    {
      title: '',
      dataIndex: '',
      valueType: 'divider',
      formItemProps: {
        wrapperCol: { span: 22 },
      },
      fieldProps: {
        children: '企业单笔消费限额',
        orientation: 'left',
      },
    },
    {
      title: '',
      dataIndex: 'mealLimit',
      formItemProps: {
        labelCol: { span: 0 },
        wrapperCol: { span: 24 },
      },
      renderFormItem: () => MealFormListLimit(meals, selectedMealTypes),
    },
    {
      title: '商品隐藏信息',
      dataIndex: 'hideInfo',
      valueType: 'checkbox',
      valueEnum: hideEnums,
    },
    {
      title: '是否周预览',
      dataIndex: 'weekPreview',
      rules: [{ required: true, message: '请选择是否周预览' }],
      valueType: 'radio',
      valueEnum: weekEnums,
      convertValue: v => `${v}`,
    },
  ];
  return baseColumn;
};
