import React, { useState, useRef, useEffect, useCallback } from 'react';
import { Spin, Button, Modal } from 'antd';
import { Title, WrapperContainer } from './components/CommonTemplate';
import { TaskTypeSelect } from './components/TaskTypeSelect';
import FormInformationBasic from './components/FormInformationBasic';
import FormPriceOrStock from './components/FormPriceOrStock';
import FormRuleSetting from './components/FormRuleSetting';
import FormRuleVPictures from './components/FormRuleVPictures';
import FormSettlementOthers from './components/FormSettlementOthers';
import {
  getProductDetail,
  merchantCategoryGetAll,
  merchantBrandList,
  merchantSpecList,
} from './service';
import { ServiceContext } from './context';

/**
 * 服务商品改造-商品模块
 * @param {*} router options
 * @returns ReactDOM
 */
const ServiceGoods = options => {
  const basicRef = useRef(null);
  const stockRef = useRef(null);
  const settingRef = useRef(null);
  const picturesRef = useRef(null);
  const settleOtrRef = useRef(null);

  const [pageId, setPageId] = useState(null);
  const [isEdit, setIsEdit] = useState(false); // 是否是编辑状态
  const [productType, setProductType] = useState(1); // 商品状态
  const [pageLoading, setPageLoading] = useState(false); // 页面加载状态
  const [categoryList, setCategoryList] = useState([]); // 获取三级类目
  const [brandList, setBrandList] = useState([]); // 获取商品牌
  const [specList, setSpecList] = useState([]); // 规格列表
  const [editData, setEditData] = useState({}); // 编辑保存数据
  const baseCheckList = [basicRef, stockRef, picturesRef];
  const [checkFormList, setCheckFormList] = useState(baseCheckList);

  const [specKeyList, setSpecKeyList] = useState([]); // 记录一级规格key字段

  const changeCheckList = proType => {
    const newBaseCheckList = proType === 3 ? [
      ...baseCheckList,
      settingRef,
      settleOtrRef,
    ] : baseCheckList;
    setCheckFormList(newBaseCheckList);
  }
  useEffect(() => {
    const timer = setTimeout(() => {
      checkFormList.forEach(({ current }) => {
        current.reset()
      });
      clearTimeout(timer);
    });
  }, [checkFormList])


  const productChange = task => {
    setProductType(task.type);
    changeCheckList(task.type);
  };

  const submitEvent = async () => {
    const checkPromiseList = checkFormList.map(({ current }) => current.onCheck());
    const resuslt = await Promise.all(checkPromiseList);
    console.log(resuslt);
  };

  // 编辑回显详情数据
  const getProductDetailResponse = async id => {
    try {
      const res = await getProductDetail(id);
      if (res && res.data) {
        setProductType(res.data.type);
        setEditData(res.data);
      }
    } catch (err) {
      console.log('接口调用失败！');
    }
  };

  // 获取三级类目分类数据
  const getMerchantCategory = async () => {
    const result = await merchantCategoryGetAll();
    setCategoryList(result.data || []);
  };
  // 获取商品牌数据
  const getMerchantBrandList = async () => {
    const result = await merchantBrandList();
    setBrandList(result.data || []);
  };
  // 获取规格列表
  const getMerchantSpecList = async () => {
    const result = await merchantSpecList();
    setSpecList(result.data || []);
  };

  useEffect(() => {
    (async () => {
      if (!options.visible) {
        return;
      }
      setPageId(options.id);
      setPageLoading(true);
      await getMerchantCategory();
      await getMerchantBrandList();
      await getMerchantSpecList();
      if (options.id) {
        setIsEdit(true);
        await getProductDetailResponse(options.id);
        changeCheckList(productType);
      }
      setPageLoading(false);
    })();
  }, [options.visible, options.id]);

  const handleOk = () => {

  }

  const handleCancel = () => {
    options.onChange(false);
  }

  const onSpecCommonImgEvent = useCallback(keys => {
    setSpecKeyList(keys);
  }, [specKeyList])


  return (
    <Modal visible={options.visible} onOk={handleOk} onCancel={handleCancel} width={1366}>
      <Spin tip="正在加载..." spinning={pageLoading} delay={100}>
        <WrapperContainer>
          <ServiceContext.Provider value={{ pageId, isEdit, productType, isCard: productType === 3 }}>
            <Title title="商品类型" />
            <TaskTypeSelect productType={productType} onChange={productChange} />

            <Title title="商品基本信息编辑" />
            <FormInformationBasic
              ref={basicRef}
              editData={editData}
              categoryList={categoryList}
              brandList={brandList}
            />

            <Title title="价格与库存" />
            <FormPriceOrStock ref={stockRef} specList={specList} onSpecChange={onSpecCommonImgEvent} editData={editData} />

            <Title title="规则设置" />
            {productType === 3 && <FormRuleSetting ref={settingRef} editData={editData} />}

            <FormRuleVPictures ref={picturesRef} specKeyItem={specKeyList} editData={editData} />
            {productType === 3 && <FormSettlementOthers ref={settleOtrRef} editData={editData} />}
            <Button type="primary" onClick={submitEvent}>
              Register
            </Button>
          </ServiceContext.Provider>
        </WrapperContainer>
      </Spin>
    </Modal>
  );
};

export default ServiceGoods;
