import React, { useContext, useEffect, forwardRef, useImperativeHandle } from 'react';
import { Cascader, Form, Input, Select } from 'antd';
import { formItemLayout } from '../config';
import { ServiceContext } from '../context';

const CreateSelectOption = optionList =>
  optionList.map(brandItem => (
    <Select.Option key={brandItem.id} value={brandItem.id}>
      {brandItem.name}
    </Select.Option>
  ));

const fileterBrandOptions = (
  input,
  options,
) => options.children.includes(input);

const filterCategoryOptions = (inputValue, path) =>
  path.some(option => option.name.toLowerCase().indexOf(inputValue.toLowerCase()) > -1);

const FormInformationBasic = forwardRef((props, ref) => {
  const { editData, categoryList, brandList } = props;
  const [form] = Form.useForm();

  const onCheck = async () => {
    try {
      const values = await form.validateFields();
      return values;
    } catch (errorInfo) {
      return null;
    }
  };

  useImperativeHandle(ref, () => ({
    onCheck,
    reset: form.resetFields,
  }));
  const customer = useContext(ServiceContext);
  useEffect(() => {
    if (customer.isEdit) {
      if (!Object.keys(editData).length) return;
      const initValues = {
        brandId: editData.brandId,
        supplierId: editData.supplierId,
        name: editData.name,
        categoryId: [editData.firstCategoryId, editData.secondCategoryId, editData.thirdCategoryId],
        description: null,
      };
      if (customer.isCard) {
        const { serviceItem } = editData.skuList[0];
        initValues.description = (serviceItem || {}).description || null;
      }
      form.setFieldsValue(initValues);
    }
  }, [customer.isEdit, editData]);

  const onFinish = values => {
    console.log('Received values of form: ', values);
  };

  return (
    <Form
      {...formItemLayout}
      form={form}
      name="register"
      onFinish={onFinish}
      initialValues={{
        brandId: null,
        supplierId: '',
        name: '',
        categoryId: [],
        description: '',
      }}
      scrollToFirstError
    >
      <Form.Item
        name="brandId"
        label="供应商名称"
        rules={[{ required: true, message: '请选择供应商名称!' }]}
      >
        <Select showSearch placeholder="请选择供应商名称" filterOption={fileterBrandOptions}>
          {CreateSelectOption(brandList)}
        </Select>
      </Form.Item>
      <Form.Item
        name="name"
        label="商品名称"
        rules={[{ required: true, message: '请输入商品名称!', whitespace: true }]}
      >
        <Input placeholder="请输入商品名称" />
      </Form.Item>
      <Form.Item
        name="categoryId"
        label="商品类目"
        rules={[{ type: 'array', required: true, message: '请输入商品类目!' }]}
      >
        <Cascader
          placeholder="请选择商品类目！"
          showSearch={{ filter: filterCategoryOptions }}
          fieldNames={{ label: 'name', value: 'id', children: 'children' }}
          options={categoryList}
        />
      </Form.Item>

      {customer.isCard ? <Form.Item
        name="description"
        label="描述"
        rules={[{ required: true, message: '请输入描述！' }]}
      >
        <Input.TextArea showCount maxLength={100} placeholder="请输入描述！" />
      </Form.Item> : null}
    </Form>
  );
});

export default FormInformationBasic;
