import React, { useState, useRef, useEffect, useCallback } from 'react';
import { Spin, Button, Modal, message, notification } from 'antd';
import { ConsoleSqlOutlined } from '@ant-design/icons';
import { Title, WrapperContainer } from './components/CommonTemplate';
import { TaskTypeSelect } from './components/TaskTypeSelect';
import FormInformationBasic from './components/FormInformationBasic';
import FormPriceOrStock from './components/FormPriceOrStock';
import FormRuleSetting from './components/FormRuleSetting';
import FormRuleVPictures from './components/FormRuleVPictures';
import FormSettlementOthers from './components/FormSettlementOthers';
import FormAttr from './components/FormAttr';
import {
  merchantBrandList,
  merchantSpecList,
  afterSalesAddrsPage,
  merchantgetJdPicList,
  supplierListQuery,
  shopGetBySupplierId,
  merchantProductAdd,
  merchantProductEdit,
  getByProductType,
} from './service';
import { isUrl, filterSendData, clearCurrent } from './utils';
import { ServiceContext } from './context';
import { GOOD_MANAGE } from '@/../config/permission.config';

/**
 * 服务商品改造-商品模块
 * @param {*} router options
 * @returns ReactDOM
 */
const ServiceGoods = options => {
  const { SourceData, categoryList, virtualCategoryList, specListData, permissions } = options;
  const canAddService = permissions[GOOD_MANAGE.ADD_SERVICE_GOODS];
  const canAddNormal = permissions[GOOD_MANAGE.ADD_NORMAL_GOODS];

  const basicRef = useRef(null);
  const stockRef = useRef(null);
  const settingRef = useRef(null);
  const picturesRef = useRef(null);
  const settleOtrRef = useRef(null);
  const attrRef = useRef(null);

  const [pageId, setPageId] = useState(null);
  const [categoryIds, setCategoryIds] = useState([]); // 商品品类ID
  const [isEdit, setIsEdit] = useState(false); // 是否是编辑状态
  const [productType, setProductType] = useState(1); // 商品状态 canAddNormal ? 1 : 4
  const [pageLoading, setPageLoading] = useState(false); // 页面加载状态
  const [afterAddressList, setAfterAddressList] = useState([]); // 售后地址
  const [supplierIdList, setSupplierIdList] = useState([]); // 适用们店列表

  const [brandList, setBrandList] = useState([]); // 获取商品牌
  const [specList, setSpecList] = useState([]); // 规格列表
  const [editData, setEditData] = useState({}); // 编辑保存数据
  const [newCategoryList, setNewCategoryList] = useState({});
  // const [shopList, setShopList] = useState([]); // 供应商列表
  const [checkFormList] = useState([
    basicRef,
    attrRef,
    stockRef,
    settingRef,
    settleOtrRef,
    picturesRef,
  ]);

  const [specKeyList, setSpecKeyList] = useState([]); // 记录一级规格key字段

  const resetForm = () => clearCurrent(checkFormList).forEach(({ current }) => current.reset());

  const productChange = task => {
    setProductType(task.type);
    const timer = setTimeout(() => {
      resetForm();
      clearTimeout(timer);
    }, 1000);
  };

  const handleCancel = refresh => {
    setPageId(null);
    setIsEdit(false);
    setProductType(4); // 默认写死服务类商品
    setEditData({});
    setSpecKeyList([]);
    resetForm();
    options.onChange(false, refresh);
  };

  // 获取商品牌数据
  const getMerchantBrandList = async () => {
    if (!brandList.length) {
      const result = await merchantBrandList();
      setBrandList(result.data || []);
    }
  };
  // 获取规格列表
  const getMerchantSpecList = async () => {
    if (!specList.length) {
      const result = await merchantSpecList();
      setSpecList(result.data || []);
    }
  };

  const getAfterSalesAddrsPage = async () => {
    if (!afterAddressList.length) {
      const result = await afterSalesAddrsPage();
      setAfterAddressList(result.data.records);
    }
  };

  // const getSupplierListResponse = async () => {
  //   if (!shopList.length) {
  //     const result = await supplierListQuery();
  //     console.log('=================>result', result);
  //     setShopList(result.data);
  //   }
  // };
  const sendMerchantProductHttpRequest = async sendData => {
    try {
      setPageLoading(true);
      const sendAsyncHttpRequest = isEdit ? merchantProductEdit : merchantProductAdd;
      const addResponse = await sendAsyncHttpRequest(sendData);
      if (addResponse.data) {
        message.success(`${isEdit ? '修改' : '添加'}成功!`);
        handleCancel(true);
      }
      setPageLoading(false);
    } catch (error) {
      console.log(error);
      setPageLoading(false);
    }
  };

  const shopGetBySupplierIdResponse = async () => {
    if (!supplierIdList.length) {
      const result = await shopGetBySupplierId();
      setSupplierIdList(result.data);
    }
  };

  const shopGetByProductType = async type => {
    console.log(newCategoryList[type]);
    if (!newCategoryList[type]?.length) {
      const result = await getByProductType(type);
      setNewCategoryList({
        ...newCategoryList,
        [type]: result.data || [],
      });
    }
    // console.log(result);
  };

  const submitEvent = async () => {
    const checkPromiseList = clearCurrent(checkFormList).map(({ current }) => current.onCheck());
    const resuslt = await Promise.all(checkPromiseList);
    if (!resuslt.includes(null)) {
      const params = resuslt.reduce((origin, item) => {
        const { temp, ...other } = item;
        origin[temp] = other;
        return origin;
      }, {});
      const sendData = filterSendData(productType, params);
      if (isEdit) {
        sendData.id = pageId;
      }
      sendMerchantProductHttpRequest(sendData);
    }
  };

  const getMerchantgetJdPicList = async params => {
    const result = await merchantgetJdPicList(params);
    if (result) {
      const { detailImageList, imageList } = picturesRef.current.getFieldsValue();
      const detailList = result.detailList || [];
      const newImageList = imageList[result.firstSpecValue];
      const carouseList = result.carouseList || [];
      imageList[result.firstSpecValue] = newImageList
        ? [...newImageList, ...carouseList]
        : carouseList;
      picturesRef.current.setFieldsValue({
        // [`imageList[${data.firstSpecValue}]`]: this.state.colorImg[data.firstSpecValue],
        imageList,
        detailImageList: [...detailImageList, ...detailList],
      });
    }
  };

  useEffect(() => {
    (async () => {
      if (!options.visible) {
        return;
      }
      setPageLoading(true);
      await shopGetBySupplierIdResponse();
      // await getSupplierListResponse();
      await getMerchantBrandList();
      await getAfterSalesAddrsPage();
      await getMerchantSpecList();
      if (Object.keys(SourceData).length) {
        setEditData(SourceData);
        setPageId(SourceData.id);
        setProductType(SourceData.productType);
        setCategoryIds(SourceData.infoMation.categoryId || []);

        // changeCheckList(SourceData.productType);
        setIsEdit(true);
      }
      console.log('111 :>> ', 111);
      setPageLoading(false);
    })();
  }, [SourceData]);

  useEffect(() => {
    if (options.visible) {
      shopGetByProductType(productType);
    }
  }, [productType, options.visible]);

  const onSpecCommonImgEvent = useCallback(
    keys => {
      setSpecKeyList(keys);
    },
    [specKeyList],
  );

  const onEventBus = (event, params) => {
    if (event === 'cloneImg') {
      if (!isUrl(params.skuLink)) {
        notification.open({
          message: '提示',
          description: '请输入正确的URL！',
        });
        return;
      }
      getMerchantgetJdPicList({
        firstSpecId: params.firstSpecId,
        firstSpecValue: params.firstSpecValue,
        secondSpecId: params.secondSpecId,
        secondSpecValue: params.secondSpecValue,
        skuLink: params.skuLink,
        jdSkuInfoUrl: params.skuLink,
      });
    }
  };
  const onCategoryChange = e => {
    setCategoryIds(e);
  };

  const providerValue = {
    pageId,
    isEdit,
    productType,
    canAddService, // 是否可以添加服务商品(电子卡券)
    canAddNormal, // 是否可以添加实物商品
    isCard: productType === 4,
    isService: SourceData.state && SourceData.state !== 4,
    isJDGoods: isEdit && SourceData.pageProductType && +SourceData.pageProductType !== 1,
    onEventBus,
  };

  return (
    <Modal
      title={isEdit ? '修改商品' : '新增商品'}
      visible={options.visible}
      onCancel={() => handleCancel()}
      destroyOnClose
      width={1050}
      maskClosable={false}
      keyboard={false}
      footer={[
        <Button key="submit" type="primary" loading={pageLoading} onClick={submitEvent}>
          提交
        </Button>,
        <Button key="back" onClick={() => handleCancel()}>
          返回
        </Button>,
      ]}
    >
      <Spin tip="正在加载..." spinning={pageLoading} delay={100}>
        <WrapperContainer>
          <ServiceContext.Provider value={providerValue}>
            <Title title="商品类型" />
            <TaskTypeSelect productType={productType} onChange={productChange} />

            <Title title="商品基本信息编辑" />
            <FormInformationBasic
              ref={basicRef}
              editData={editData.infoMation}
              newCategoryList={newCategoryList}
              categoryList={categoryList}
              virtualCategoryList={virtualCategoryList}
              brandList={brandList}
              afterAddressList={afterAddressList}
              specListData={specListData}
              onCategoryChange={onCategoryChange}
            />

            {[1, 2].includes(productType) && [
              <Title title="商品属性" />,
              <FormAttr ref={attrRef} categoryIds={categoryIds} initData={editData} />,
            ]}

            <Title title="价格与库存" />
            <FormPriceOrStock
              ref={stockRef}
              specList={specList}
              onSpecChange={onSpecCommonImgEvent}
              editData={editData.infoSpecData}
              skuList={editData.skuList}
            />

            <Title title="规则设置" />
            {productType === 4 && (
              <FormRuleSetting
                ref={settingRef}
                editData={editData.serviceItem}
                supplierIdList={supplierIdList}
              />
            )}

            <FormRuleVPictures
              ref={picturesRef}
              specKeyItem={specKeyList}
              editData={editData.infoImageData}
            />
            {productType === 4 && (
              <FormSettlementOthers ref={settleOtrRef} editData={editData.settlementItem} />
            )}
          </ServiceContext.Provider>
        </WrapperContainer>
      </Spin>
    </Modal>
  );
};

export default ServiceGoods;
