import React, { useEffect, useState } from 'react';
import {
  Button,
  Divider,
  Form,
  Input,
  InputNumber,
  Checkbox,
  Radio,
  Col,
  Row,
  Modal,
  Select,
  message,
  notification,
} from 'antd';
import { ExclamationCircleOutlined } from '@ant-design/icons';
import moment from 'moment';
import { jsonToArray } from '@/utils/utils';
import { layout, mealType, boolOptions, hideOptions, mealSections } from '../staticData/index';
import {
  apiEnterpriseInfo,
  apiNewEnterprise,
  apiEditEnterprise,
  apiEnterprisePickSelf,
} from '../service';
import MealSection from './MealSection';
import MealLimit from './MealLimit';

const CheckboxGroup = Checkbox.Group;
const RadioGroup = Radio.Group;
const { confirm } = Modal;
const CustomerInfo = props => {
  const [form] = Form.useForm();
  const [meals, setMeals] = useState([]);
  const [selectedMealTypes, setSelectedMealTypes] = useState([]);
  const [pickSelfList, setPickSelfList] = useState([]);

  /*
   *  7. 表单关闭
   */
  const closeModal = index => {
    if (props.onClose && typeof props.onClose === 'function') {
      props.onClose(!!index);
    }
    setSelectedMealTypes([]);
    setMeals({});
    form.setFieldsValue({
      mealLimit: [],
    });
  };

  /*
   *  6. 表单提交
   */
  const submitForm = async params => {
    let api = apiNewEnterprise;
    if (props.id) {
      params.id = props.id;
      api = apiEditEnterprise;
    }
    const resp = await api(params);
    if (resp && resp.data) {
      // 保存成功后刷新列表
      closeModal(1);
      notification.success({ message: '保存成功！' });
    }
  };

  /*
   *  5. 表单提交数据处理及校验
   */
  const validateForm = async () => {
    const res = await form.validateFields();
    const params = {
      hideImage: 0,
      hidePrice: 0,
      ...res,
    };
    const arr = [];
    res.mealTimePeriod.forEach(item => {
      if (item && meals[item.mealPeriodType]) {
        const obj = { ...item };
        obj.beginTime = moment(obj.time[0]).format('HH:mm');
        obj.endTime = moment(obj.time[1]).format('HH:mm');
        delete obj.time;
        arr.push(obj);
      }
    });
    params.mealTimePeriod = arr;
    if (res.hideInfo && res.hideInfo.length) {
      params.hidePrice = res.hideInfo.includes('hidePrice') ? 1 : 0;
      params.hideImage = res.hideInfo.includes('hideImage') ? 1 : 0;
      delete params.hideInfo;
    }
    const limits = [];
    // 处理限额
    Object.keys(res.mealLimit).forEach(item => {
      const mealPeriodType = item.replace('limit', '');
      if (meals[mealPeriodType]) {
        const json = {
          mealPeriodType,
          limit: [],
        };
        Object.keys(res.mealLimit[item]).forEach(t => {
          if (selectedMealTypes.includes(t)) {
            json.limit.push({
              mealType: t,
              limit: res.mealLimit[item][t],
            });
          }
        });
        limits.push(json);
      }
    });
    params.mealLimit = limits;
    console.log(params, '.....');
    // submitForm(params);
  };

  /*
   *  4. 表单交互逻辑
   */
  // 校验时间
  const checkTime = (arr, curren, curName) => {
    let valid = false;
    arr.forEach(item => {
      if (curren < item.endTime) {
        valid = true;
        const name = meals[item.mealPeriodType];
      }
    });
    return valid;
  };
  const validateMeals = () => {
    const { mealTimePeriod = [] } = form.getFieldValue();
    const arr = [];
    let validTime = false;
    mealTimePeriod.forEach(item => {
      if (item && meals[item.mealPeriodType]) {
        const obj = { ...item };
        obj.beginTime = moment(obj.time[0]).format('HH:mm');
        obj.endTime = moment(obj.time[1]).format('HH:mm');
        delete obj.time;
        if (checkTime(arr, obj.beginTime, meals[item.mealPeriodType])) {
          validTime = true;
        }
        arr.push(obj);
      }
    });
    if (validTime) {
      // eslint-disable-next-line prefer-promise-reject-errors
      return Promise.reject('时间段不能交叉！');
    }
    return Promise.resolve();
  };
  // 风险提示
  const checkConfirm = () => {
    const mt = '';
    return new Promise((resolve, reject) => {
      confirm({
        title: '风险提示',
        icon: <ExclamationCircleOutlined />,
        content: `确定关闭${mt}餐品类型？此餐品类型下关联的商户及商品将一并删除，不可逆请谨慎操作！`,
        onOk() {
          resolve(1);
        },
        onCancel() {
          reject(new Error());
        },
      });
    });
  };
  // 改变餐品类型 (选自助餐必选外卖)
  const onChangeMealType = async ms => {
    try {
      // 编辑时，取消餐段，提示确认
      if (props.id && ms.length < selectedMealTypes.length) {
        await checkConfirm();
      }
      // 取消外卖，必须取消自助餐
      if (selectedMealTypes.includes('1') && !ms.includes('1')) {
        ms = ms.filter(item => item !== '2');
      }

      // 选择自助餐，必须选择外卖
      if (!selectedMealTypes.includes('1') && ms.includes('2')) {
        ms.push('1');
      }

      form.setFieldsValue({
        mealType: ms,
      });
      setSelectedMealTypes(ms);
    } catch {
      form.setFieldsValue({
        mealType: selectedMealTypes,
      });
    }
  };
  // 改变餐段
  const onChangeMealSection = e => {
    console.log(e);
    const { id, checked, label } = e.target;
    const values = Object.assign({}, meals);
    if (checked) {
      values[id] = label;
    } else {
      delete values[id];
    }
    // 判断餐段配置是否都没选
    if (Object.keys(values).length === 0) {
      form.setFieldsValue({
        mealTimePeriod: [],
      });
    }
    setMeals(values);
  };

  /*
   *  3.表单数据处理，
   */
  const processMealTimePeriod = mealTimePeriod => {
    const mealTimePeriodMap = {};
    const mealTimePeriodArr = Object.keys(mealSections).map(() => ({}));

    mealTimePeriod.forEach(item => {
      const { mealPeriodType, beginTime, endTime } = item;
      // 数据模型转换-餐段配置,转为 {餐段:餐段名称}
      // 把mealTimePeriod按mealPeriodType转为map
      const mealPeriodName = mealSections[mealPeriodType];
      mealTimePeriodMap[mealPeriodType] = mealPeriodName;

      // 数据模型转换-餐段和时间配置, [{餐段, time}, {}, {}]
      const index = Object.keys(mealSections).indexOf(`${mealPeriodType}`);
      if (index > -1) {
        mealTimePeriodArr[index] = {
          mealPeriodType: `${mealPeriodType}`,
          time: [
            moment(`${moment().format('YYYY-MM-DD')} ${beginTime}`),
            moment(`${moment().format('YYYY-MM-DD')} ${endTime}`),
          ],
        };
      }
    });
    return { mealTimePeriodMap, mealTimePeriodArr };
  };
  // 数据模型转换-消费限额, 转为{餐段: {餐品类型: 限额}}
  const processMealLimit = mealLimit => {
    const mealLimitMap = {};
    mealLimit.forEach(item => {
      const { mealPeriodType, limit } = item;

      mealLimitMap[`limit${mealPeriodType}`] = limit.reduce(
        (acc, t) => ({
          ...acc,
          [t.mealType]: t.limit,
        }),
        {},
      );
    });
    return mealLimitMap;
  };

  const processData = data => {
    const {
      hideImage,
      hidePrice,
      id,
      name,
      mealLimit,
      mealTimePeriod = [],
      mealType: type,
      weekPreview,
      endOrderTime,
    } = data;

    const formData = {
      id,
      name,
      weekPreview,
      mealType: type?.map(item => `${item}`) ?? [],
      hideInfo: [],
      endOrderTime,
    };

    if (+hidePrice) {
      formData.hideInfo.push('hidePrice');
    }
    if (+hideImage) {
      formData.hideInfo.push('hideImage');
    }

    const { mealTimePeriodMap, mealTimePeriodArr } = processMealTimePeriod(mealTimePeriod);
    const mealLimitMap = processMealLimit(mealLimit);

    formData.mealTimePeriod = mealTimePeriodArr;
    formData.mealLimit = mealLimitMap;
    formData.mealTimePeriodMap = mealTimePeriodMap;
    return formData;
  };

  /*
   *  2. 表单数据获取
   */
  // 获取自提点列表
  const getPickSelf = async () => {
    const res = await apiEnterprisePickSelf({});
    if (res && res.data && res.data.records) {
      const data = res.data.records;
      setPickSelfList(
        data.map(item => ({
          value: item.id,
          label: item.pickselfName,
        })),
      );
    }
  };
  // 获取表单信息
  const getInfo = async () => {
    const res = await apiEnterpriseInfo(props.id);
    if (res && res.data) {
      const formData = processData(res.data);
      setSelectedMealTypes(formData.mealType);
      setMeals(formData.mealTimePeriodMap);
      form.setFieldsValue(formData);
    }
  };

  useEffect(() => {
    if (props.visible) {
      if (props.id) {
        getInfo();
      }
    }
  }, [props.visible]);

  /*
   *  1. 表单结构
   */
  const MealLimitsFormList = item => (
    <Form.List name="mealLimit" key={item.name}>
      {mealLimitsFields => (
        <>
          {Object.keys(meals).map(meal => (
            <Form.Item
              key={`${mealSections[meal]}`}
              label={`${mealSections[meal]}订单`}
              required
              wrapperCol={{ span: 20 }}
            >
              <Form.List
                name={`limit${meal}`}
                key={`${meal}limit`}
                required
                wrapperCol={{ span: 20 }}
              >
                {mealLimitsFieldsList => (
                  <Row key={`row${meal}`}>
                    {selectedMealTypes.map((t, i) => (
                      <Col span={7} offset={i ? 1 : 0} key={t}>
                        <MealLimit value={t} label={mealType[t]} name={`${t}`} />
                      </Col>
                    ))}
                  </Row>
                )}
              </Form.List>
            </Form.Item>
          ))}
        </>
      )}
    </Form.List>
  );
  const formItem = [
    {
      type: 'Form.Item',
      label: '企业名称',
      name: 'name',
      wrapperCol: { span: 20 },
      rules: [{ required: true, message: '请输入企业名称' }],
      component: <Input placeholder="请输入企业名称" />,
    },
    {
      type: 'Form.Item',
      disabled: props.id,
      label: '企业取餐点',
      wrapperCol: { span: 20 },
      name: 'pickSelfIds',
      rules: [{ required: true, message: '请选择企业取餐点' }],
      component: (
        <Select
          options={pickSelfList}
          mode="multiple"
          showSearch
          filterOption={(v, option) =>
            (option?.label ?? '').toLowerCase().includes(v.toLowerCase())
          }
        />
      ),
    },
    {
      type: 'Form.Item',
      label: '企业截止时间',
      wrapperCol: { span: 20 },
      name: 'endOrderTime',
      rules: [{ required: true, message: '请输入企业截止时间' }],
      component: <InputNumber min={0} max={600} addonAfter="分钟" />,
      extra: <span>企业员工下单的截至时间，仅支持正整数，单位为分钟。</span>,
    },
    {
      type: 'Form.Item',
      label: '餐品类型',
      wrapperCol: { span: 20 },
      name: 'mealType',
      rules: [{ required: true, message: '请选择餐品类型' }],
      component: <CheckboxGroup options={jsonToArray(mealType)} onChange={onChangeMealType} />,
    },
    {
      type: 'Form.Item',
      label: '餐段配置',
      wrapperCol: { span: 12 },
      name: 'mealTimePeriod',
      rules: [{ required: true, message: '请选择餐段配置' }],
      component: (
        <MealSection meals={meals} validateMeals={validateMeals} onChange={onChangeMealSection} />
      ),
    },
    {
      name: 'Divider',
      type: 'Divider',
      component: '企业单笔消费限额',
    },
    {
      name: 'Form.List',
      type: 'Form.List',
      component: '',
    },
    {
      type: 'Form.Item',
      label: '商品隐藏信息',
      wrapperCol: { span: 20 },
      name: 'hideInfo',
      rules: [{ required: true, message: '请选择商品隐藏信息' }],
      component: <CheckboxGroup options={hideOptions} />,
    },
    {
      type: 'Form.Item',
      label: '是否周预览',
      wrapperCol: { span: 20 },
      name: 'weekPreview',
      rules: [{ required: true, message: '请选择是否周预览' }],
      component: <RadioGroup options={boolOptions} />,
    },
  ];

  return (
    <Modal
      title="企业客户信息"
      open={props.visible}
      destroyOnClose
      maskClosable={false}
      width="900px"
      onOk={validateForm}
      onCancel={() => closeModal(0)}
    >
      <Form name="basicInfo" initialValues={{ mealTimePeriod: [] }} {...layout} form={form}>
        {formItem.map((item, index) => {
          if (item.type === 'Divider' && !item.disabled) {
            return (
              <Divider key={item.name} orientation="left" plain>
                {item.component}
              </Divider>
            );
          }
          if (item.type === 'Form.Item' && !item.disabled) {
            return (
              <Form.Item
                key={item.name}
                label={item.label ?? ''}
                name={item.name}
                rules={item.rules ?? []}
                wrapperCol={item.wrapperCol}
                extra={item.extra ?? ''}
              >
                {item.component}
              </Form.Item>
            );
          }
          if (item.type === 'Form.List' && !item.disabled) {
            return MealLimitsFormList(item);
          }
          return null;
        })}
      </Form>
    </Modal>
  );
};
export default CustomerInfo;
