import React, {
  useState,
  useContext,
  useEffect,
  forwardRef,
  useImperativeHandle,
  useRef,
} from 'react';
import { Cascader, Form, Input, Select, Popover, Button, Checkbox, Divider, Modal } from 'antd';
import { formItemLayout } from '../config';
import { ServiceContext } from '../context';
import { debounce } from '@/utils/utils';
import AddMenusModal from './AddMenusModal';
import UploadImage from './UploadImage';
import { apiShopIds, apiQueryShopList, getByProductType } from '../service';

const CreateSelectOption = optionList =>
  optionList.map(brandItem => (
    <Select.Option key={brandItem.id} value={brandItem.id}>
      {brandItem.name}
    </Select.Option>
  ));

const fileterBrandOptions = (input, options) => options.children.includes(input);
const filterCategoryOptions = (inputValue, path) =>
  path.some(option => option.name.toLowerCase().indexOf(inputValue.toLowerCase()) > -1);

const FormInformationBasic = forwardRef((props, ref) => {
  const {
    editData,
    newCategoryList,
    categoryList,
    virtualCategoryList,
    brandList,
    afterAddressList,
    specListData,
    shopGetByProductType,
  } = props;
  const [form] = Form.useForm();
  const [noreBrandList, setNoreBrandList] = useState([]);
  const customer = useContext(ServiceContext);
  const childAddMenusModalRef = useRef(null);
  const [takeawayImageList, setTakeawayImageList] = useState([]);
  const [shopIds, setShopIds] = useState([]);
  const [shopList, setShopList] = useState([]);
  const [isEditTakeaway, setIsEditTakeaway] = useState(false);

  const onCheck = async () => {
    try {
      const values = await form.validateFields();
      const specs = [];
      // 非服务类商品过滤specs数据
      if (!customer.isCard) {
        specListData.forEach(item => {
          const specValues = values[item.specId];
          // 判断是否包含改参数
          if (specValues) {
            specs.push({ specId: item.specId, specValues });
          }
          delete values[item.specId];
        });
      }
      values.specs = specs;
      values.temp = 'infoMation';
      return values;
    } catch (errorInfo) {
      return null;
    }
  };

  // 自定义加入菜单
  const showModal = () => {
    childAddMenusModalRef.current.setOpen(true);
  };
  // 自定义菜单下拉
  const dropdownRender = menus => (
    <div>
      {menus}
      <Divider
        style={{
          margin: 0,
        }}
      />
      <div
        style={{
          padding: 8,
          background: '#1890ff',
          color: '#fff',
          textAlign: 'center',
        }}
        onClick={showModal}
      >
        添加自定义分组
      </div>
    </div>
  );
  const onTakeawayImageList = imgList => {
    setTakeawayImageList(imgList);
    form.setFieldsValue({
      commonImageList: imgList,
    });
  };
  const getFormValues = debounce(() => {
    const values = form.getFieldsValue();
    props.onValuesChange({ infoMation: values });
  }, 400);
  // 查询shopIds
  const queryShopIds = async () => {
    if (!shopIds.length) {
      const result = await apiShopIds();
      setShopIds(result.data || []);
    }
  };
  // 查询分组列表
  const queryShopList = async params => {
    if (!shopList.length) {
      const result = await apiQueryShopList(params);
      setShopList(result.data || []);
    }
  };
  const onChangeShopId = async e => {
    if (e) {
      setShopList([]);
      queryShopList({ shopId: e });
    }
  };
  useImperativeHandle(ref, () => ({
    onCheck,
    reset: form.resetFields,
  }));
  useEffect(() => {
    queryShopIds();
  }, [customer.productType]);

  useEffect(() => {
    const noreList = (brandList || []).filter(item => item.name === '虚拟商品');
    setNoreBrandList(noreList);
  }, [brandList]);

  useEffect(() => {
    if (!editData) return;
    if (editData.productType === 5 && editData.name) {
      shopGetByProductType(5);
      queryShopList({ shopId: editData.productRefShopId });
      setIsEditTakeaway(true);
    }
    form.setFieldsValue(editData);
    onTakeawayImageList(editData.commonImageList);
  }, [customer.isEdit, editData]);

  return (
    <Form
      {...formItemLayout}
      form={form}
      name="register"
      initialValues={{
        brandId: null,
        name: '',
        categoryId: [],
        description: '',
        productRefShopId: [],
        storageRackIds: [],
      }}
      scrollToFirstError
      onValuesChange={getFormValues}
    >
      <Popover content={form.getFieldValue('name')} trigger="hover">
        <Form.Item
          key="name"
          name="name"
          label="商品名称"
          rules={[
            { required: true, min: 2, message: '请输入最少两个字符的商品名称', whitespace: true },
          ]}
        >
          <Input placeholder="请输入商品名称" disabled={customer.isDisabled || isEditTakeaway} />
        </Form.Item>
      </Popover>
      {customer.isTakeawayService && (
        <Form.Item
          name="productRefShopId"
          label="所属门店"
          rules={[{ required: true, message: '请选择所属门店' }]}
        >
          <Select
            fieldNames={{ label: 'name', value: 'id' }}
            filterOption={fileterBrandOptions}
            placeholder="请选择所属门店"
            options={shopIds}
            onChange={onChangeShopId}
          ></Select>
        </Form.Item>
      )}
      {/* 菜单分组 */}
      {customer.isTakeawayService && (
        <Form.Item
          name="storageRackIds"
          label="菜单分组"
          rules={[{ type: 'array', required: true, message: '请输入菜单分组' }]}
        >
          <Cascader
            placeholder="请选择菜单分组"
            disabled={customer.isEdit && customer.isNormal}
            showSearch={{ filter: filterCategoryOptions }}
            fieldNames={{ label: 'name', value: 'id', children: 'children' }}
            // onChange={props.onCategoryChange}
            options={shopList}
            dropdownRender={dropdownRender}
          />
        </Form.Item>
      )}
      {/* 新增菜单分组弹框 */}
      <AddMenusModal ref={childAddMenusModalRef} queryShopList={queryShopList} />
      <Form.Item
        name="categoryId"
        label="商品类目"
        rules={[{ type: 'array', required: true, message: '请输入商品类目' }]}
      >
        <Cascader
          placeholder="请选择商品类目"
          disabled={customer.isEdit && customer.isNormal}
          showSearch={{ filter: filterCategoryOptions }}
          fieldNames={{ label: 'name', value: 'id', children: 'children' }}
          onChange={props.onCategoryChange}
          options={newCategoryList[customer.productType]}
        />
      </Form.Item>
      {/* 外卖-商品图片 */}
      {customer.isTakeawayService && (
        <Form.Item
          name="commonImageList"
          label="商品图片"
          extra="支持.jpg/png格式图片，建议单张比例1:1，大小200kb左右，最多可以上传5张"
          rules={[
            {
              required: true,
              type: 'array',
              message: '请上传商品图片',
            },
          ]}
        >
          <UploadImage
            name="takeawayImageList"
            limit={5}
            disabled={customer.isDisabled}
            pictures={takeawayImageList}
            setPictureList={list => onTakeawayImageList(list)}
          />
        </Form.Item>
      )}

      {!customer.isTakeawayService && !customer.isCard && (
        <Form.Item
          name="brandId"
          label="商品品牌"
          rules={[{ required: true, message: '请选择商品品牌' }]}
          extra="若需新增品牌请联系业务员"
        >
          <Select
            showSearch
            disabled={customer.isDisabled}
            placeholder="请选择商品品牌"
            filterOption={fileterBrandOptions}
          >
            {customer.productType === 2
              ? CreateSelectOption(noreBrandList)
              : CreateSelectOption(brandList)}
          </Select>
        </Form.Item>
      )}

      {!customer.isTakeawayService && !customer.isCard && (
        <Form.Item
          name="character"
          label="商品卖点"
          rules={[{ required: true, message: '请输入商品卖点!', whitespace: true }]}
        >
          <Input
            maxLength={50}
            placeholder="卖点最优可填写3个词,12个字。前后用空格加竖杠分隔,例: 莹莹剔透 | 粒粒优选 | 易煮易熟"
          />
        </Form.Item>
      )}

      {!customer.isTakeawayService && !customer.isCard && (
        <Form.Item name="afterAddressId" label="售后地址">
          <Select showSearch placeholder="请选择售后地址" filterOption={fileterBrandOptions}>
            {(afterAddressList || []).map(item => (
              <Select.Option key={item.id} value={item.id}>
                {item.addressName}
              </Select.Option>
            ))}
          </Select>
        </Form.Item>
      )}

      {!customer.isTakeawayService &&
        !customer.isCard &&
        specListData.map((item, index) => (
          <Form.Item name={item.specId} key={item.specId} label={item.specName}>
            <Checkbox.Group options={item.specValues} />
          </Form.Item>
        ))}

      {/* <Form.Item
        name="brandId"
        label="供应商名称"
        rules={[{ required: true, message: '请选择供应商名称!' }]}
      >
        <Select showSearch placeholder="请选择供应商名称" filterOption={fileterBrandOptions}>
          {CreateSelectOption(brandList)}
        </Select>
      </Form.Item> */}

      {customer.isCard ? (
        <Form.Item
          key="description"
          name="description"
          label="描述"
          rules={[{ required: true, message: '请输入描述,100字以内' }]}
        >
          <Input.TextArea showCount maxLength={100} placeholder="请输入描述, 100字以内" />
        </Form.Item>
      ) : null}
    </Form>
  );
});

export default FormInformationBasic;
