// import fileSaver from 'file-saver';
import request from '@/utils/request';
import config from '../../../config/env.config';
import { stringify } from 'qs';
import { saveAs } from 'file-saver';
import { format } from 'date-fns';
import _ from 'lodash';

const { goodsApi, kdspApi } = config;
const headers = {
  'Content-Type': 'application/x-www-form-urlencoded',
};

/**
 * 商品列表
 * yapi: http://yapi.quantgroups.com/project/389/interface/api/23814
 */
export async function searchList(params) {
  return request.post('/api/merchants/products/page', {
    prefix: goodsApi,
    data: stringify(params),
    headers,
    role: true,
  });
}

// 新增商品
export async function addGoods(params) {
  return request.post('/api/merchants/products/add', {
    prefix: goodsApi,
    data: params,
  });
}

// 编辑商品

export async function editGoods(params) {
  return request.post('/api/merchants/products/edit', {
    prefix: goodsApi,
    data: params,
  });
}

// 获取商品品牌
export async function getBrandList() {
  return request.post('/product/brand/api/merchant/list', {
    prefix: goodsApi,
  });
}

// 获取类目关联属性
export async function apiGetAttribute(categoryId) {
  const data = await request.get(
    `/api/kdsp/category/template/ref/attribute/detail?categoryId=${categoryId}`,
    {
      prefix: goodsApi,
    },
  );
  return data;
}
// 编辑--获取详情
export async function spuDetail(params) {
  return request.post('/api/merchants/products/sku-detail', {
    prefix: goodsApi,
    params,
    headers,
  });
}

// 商品规格
export async function getSpecList() {
  return request.post('/api/merchants/specs/getAll', {
    prefix: goodsApi,
  });
}

// 查询图片素材
export async function getImageInfo(spuNo) {
  return request.post('/product/api/merchant/imageInfo', {
    params: { spuNo },
    prefix: goodsApi,
    headers,
  });
}

// 商品分类
export async function categoryList() {
  return request.post('/api/merchants/categories/getAllTree', {
    prefix: goodsApi,
  });
}
/**
 * 商品分类
 * type 商品类型：1-实物类，2-虚拟类，4-服务类 5 外卖
 * */
export async function apiCategoryListType(type) {
  return request.get(`/product/category/getByProductType/${type}`, {
    prefix: goodsApi,
  });
}

// 批量修改
export async function uploadFile(file) {
  const params = new FormData();
  params.append('file', file);
  const data = await request.post('/product/api/merchant/BatchUpdateStock', {
    prefix: goodsApi,
    data: params,
    notTip: true,
  });
  return data;
}

// 商品修改日志
export async function changeLog(params) {
  return request.post('/api/merchant/change/logs/page', {
    data: stringify(params),
    prefix: goodsApi,
    headers,
  });
}

// 审核详情日志
export async function productMerchantLog(productId) {
  return request.get('/api/merchants/audit/logs/list', {
    params: {
      productId,
    },
    prefix: goodsApi,
  });
}

// 拉去京东图片
export async function getJdPicList(params) {
  const { data } = await request.post('/product/api/merchant/item/getJdPicList', {
    data: stringify(params),
    prefix: goodsApi,
    headers,
  });
  return data;
}

// 图片上传
export async function uploadImg(files) {
  const params = new FormData();
  files.forEach(file => params.append('file', file));
  const data = await request.post('/image/api/merchant/upload', {
    prefix: goodsApi,
    data: params,
  });
  return data;
}

// 修改商品库存
export async function updateStock(params) {
  const data = await request.post('/product/item/api/merchant/updateStock', {
    prefix: goodsApi,
    data: stringify(params),
    headers,
  });
  if (data.businessCode === '0000') {
    return null;
  }
  return data.msg;
}

// 获取虚拟商品类目
export async function getVirtualCategory() {
  const data = await request.post('/product/category/api/merchant/getByParentId', {
    prefix: goodsApi,
    data: stringify({ id: 100018 }),
    headers,
  });
  return data;
}

// 查询商品属性列表
export async function specList() {
  return request.get('/product/api/merchant/spec', {
    prefix: goodsApi,
  });
}
// sku绑定模板
export async function getTemplate(params) {
  const data = await request.post('/api/kdsp/sku/template', {
    data: params,
    prefix: goodsApi,
  });
  return data;
}

// 不配送区域模板列表
export async function getTemplateList(params) {
  const data = await request.post('/api/kdsp/template/list', {
    data: stringify(params),
    prefix: goodsApi,
    headers,
  });
  return data;
}
// 查询供应商售后地址
export async function queryAllAfterAddress() {
  const params = {
    pageSize: 100,
    pageNo: 1,
  };
  const data = request.post('/api/merchants/aftersales/addresses/list', {
    prefix: kdspApi,
    data: stringify(_.omitBy(params, v => !v)),
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded',
    },
  });
  return data;
}

// 商品售后地址设置
export async function getAfterAddress(params) {
  const data = await request.post('/api/kdsp/updateSkuAfterAddress', {
    data: params,
    prefix: goodsApi,
  });
  return data;
}

/**
 * 商品是否可以做库存变更
 * skuIds: 多个用英文逗号隔开
 */
export const apiEnableUpdataStock = skuIds =>
  request.get(`/api/kdsp/sku/can/stockChange?skuIds=${skuIds}`, {
    prefix: goodsApi,
  });

// 商品明细导出
export const apiGoodsInfosExport = params =>
  request.post('/api/kdsp/sku/export', {
    prefix: goodsApi,
    data: params,
    responseType: 'arrayBuffer',
    headers: {
      'Content-Type': 'application/json;charset=UTF-8',
    },
  });

// 新建商品审核记录
export const apiCreateGoodsLog = params =>
  request.post('/api/kdsp/product/audit/create', {
    data: params,
    prefix: goodsApi,
  });

// 查询sku最后一条库存变更记录
export const apiQueryLastChangeLog = skuId =>
  request.get(`/api/kdsp/sku/last/stockChange/record?skuId=${skuId}`, {
    prefix: goodsApi,
  });

// 商品上下架
export const apiChangeStateGoods = async params => {
  const data = await request.post('/api/merchants/products/api/merchant/pushedOrOffline/batch', {
    data: stringify(params),
    prefix: goodsApi,
    headers,
  });
  return data;
};

// 查询sku最后一条审核记录
export const apiQueryLastAuditRecord = skuId =>
  request.get(`/api/kdsp/sku/last/audit/record?skuId=${skuId}`, {
    prefix: goodsApi,
  });

// 商品草稿详情
export const apiDraftDetail = draftId =>
  request.get(`/api/merchants/drafts/detail?id=${draftId}`, {
    prefix: goodsApi,
  });

// 删除商品草稿
export const apiDeleteDraft = draftId =>
  request.get(`/api/merchants/drafts/delete?id=${draftId}`, {
    prefix: goodsApi,
  });

// 商品草稿列表
export async function apiDraftList(data) {
  return request.post('/api/merchants/drafts/list', {
    prefix: goodsApi,
    data,
  });
}

// 批量操作
export async function apiGoodsActionBatch(data) {
  return request.post('/api/merchants/products/sku/batchOperation', {
    prefix: goodsApi,
    data,
  });
}
// 外卖商品列表
export async function apiTakeawayGoods(params) {
  return request.post('/api/merchants/products/page', {
    prefix: goodsApi,
    data: stringify(params),
    headers,
  });
}
// 外卖商品排序
// export async function apiSortTakeawayGoods(data) {
//   return request.post('/api/merchants/products/sku/batchSort', {
//     prefix: goodsApi,
//     data,
//   });
// }
/**
 * 外卖商品排序
 * yapi: http://yapi.quantgroups.com/project/389/interface/api/58174
 */
export async function apiSortTakeawayGoods(data) {
  return request.post('/api/merchants/products/sku/range/sort', {
    prefix: goodsApi,
    data,
  });
}
// 外卖商品置顶
export async function apiTopTakeawayGoods(data) {
  return request.post('/api/merchants/products/sku/storageRack/topping', {
    prefix: goodsApi,
    data,
  });
}
// 获取供应商门店列表
export async function apiSupplierShopList(params) {
  return request.get(`/api/merchants/shops/getBySupplierId?${stringify(params)}`, {
    prefix: goodsApi,
  });
}
// 分组创建（货架—创建货架）
export async function apiCreateStorage(data) {
  return request.post('/api/merchants/products/storageRack/create', {
    prefix: goodsApi,
    data,
  });
}
// 编辑分组（货架—编辑货架）
export async function apiEditStorage(data) {
  return request.post('/api/merchants/products/storageRack/edit', {
    prefix: goodsApi,
    data,
  });
}
// 分组详情（货架—货架详情）
export async function apiStorageInfo(params) {
  return request.post('/api/merchants/products/storageRack/queryByShopIdAndStorageRackId', {
    prefix: goodsApi,
    data: stringify(params),
    headers,
  });
}
// 删除分组（货架—删除货架）
export async function apiDelStorage(params) {
  return request.post('/api/merchants/products/storageRack/removeByShopIdAndId', {
    prefix: goodsApi,
    data: stringify(params),
    headers,
  });
}
// 分组排序（货架—排序货架）
export async function apiSortStorage(data) {
  return request.post('/api/merchants/products/storageRack/batchSort', {
    prefix: goodsApi,
    data,
  });
}
// 分组列表（货架—货架列表）
export async function apiStorageList(params) {
  return request.post('/api/merchants/products/storageRack/listByShopIdAndStorageRackIds', {
    prefix: goodsApi,
    data: stringify(params),
    headers,
  });
}
// 获取库存信息
export async function apiProductStock(data) {
  return request.get('/api/merchants/products/sku/getStockInfo', {
    prefix: goodsApi,
    params: data,
  });
}

/**
 * 获取竞价详情
 * yApi: http://yapi.quantgroups.com/project/389/interface/api/67169
 * * */
export async function apiProductBiddingInfo(params) {
  return request.get('/api/merchants/products/bidding/detail', {
    prefix: goodsApi,
    params,
  });
}

/**
 * 供应商更新商品供货价库存
 * yApi: http://yapi.quantgroups.com/project/389/interface/api/67139
 * * */
export async function apiProductBiddingUpdate(params) {
  return request.post('/api/merchants/products/bidding/edit', {
    prefix: goodsApi,
    data: params,
    duration: null,
    notTip: true,
  });
}

/**
 * 可竞价sku列表
 * yApi: http://yapi.quantgroups.com/project/389/interface/api/67164
 * * */
export async function apiBiddingList(params) {
  return request.post('/api/merchants/products/bidding/page', {
    prefix: goodsApi,
    data: stringify(params),
    headers,
  });
}

/**
 * 导入竞价商品信息
 * yApi: http://yapi.quantgroups.com/project/389/interface/api/45896
 * * */
export async function apiUploadGoodsFile(file) {
  const params = new FormData();
  params.append('file', file);
  params.append('type', 6);
  const data = await request.post('/api/merchants/importFile/excel', {
    data: params,
    prefix: goodsApi,
  });
  return data;
}
/**
 * 下载竞价商品模版
 * yApi: http://yapi.quantgroups.com/project/389/interface/api/67269
 * * */
export async function apiDownBiddingTemplate(params) {
  const productCategoryId = params?.productCategoryId || [];
  params.productCategoryId =
    (productCategoryId.length && productCategoryId[productCategoryId.length - 1]) || '';
  const res = await request.post('/api/merchants/products/bidding-template/export', {
    data: stringify(_.omitBy(params, v => v === undefined && v === null && v === '')),
    headers,
    prefix: goodsApi,
    responseType: 'arrayBuffer',
  });
  const blob = new Blob([res]);
  saveAs(blob, `自营商品供货价更新表-${format(new Date(), 'yyyy-MM-dd')}.xlsx`);
}
