/* eslint-disable no-shadow */
import { Button, Upload, notification, Spin } from 'antd';
import React, { useRef, useEffect, useState } from 'react';
import { PageHeaderWrapper } from '@ant-design/pro-layout';
import ProTable from '@ant-design/pro-table';
import { saveAs } from 'file-saver';
import { format } from 'date-fns';
import { columns, toolBarRender } from './data';
import { query, confirmSelfBill, downloadSelfBillDetail } from './service';

const ReconciliationQuery = () => {
  const actionRef = useRef();
  const [selectedRowKeys, setselectedRowKeys] = useState([]);
  const [selectedRows, setselectedRows] = useState([]);
  const [loading, setloading] = useState(false);

  const reload = () => {
    if (actionRef.current) {
      actionRef.current.reload();
    }
  };

  // 检查是否是同类型的
  const checkedBillType = data => {
    const billType = data[0]?.billType;
    const sameData = data.filter(item => item.billType === billType);
    if (data.length === sameData.length && sameData.length < 32) {
      return true;
    }
    if (sameData.length > 31) {
      notification.error({ message: '最多选择31条数据账单' });
      return false;
    }
    notification.error({ message: '不同账单类型不可批量操作' });
    return false;
  };

  // 检查是否是确认账单
  const checkedStatus = data => {
    const sameData = data.filter(item => item.status === 1);
    if (sameData.length > 0 && data.length === sameData.length) {
      return true;
    }
    if (!sameData.length) {
      notification.error({ message: '请选择数据账单' });
      return false;
    }
    notification.error({ message: '只有待确认账单可以确认' });
    return false;
  };

  const onSelectChange = (selectedRowKeys, selectedRows) => {
    if (checkedBillType(selectedRows)) {
      setselectedRowKeys(selectedRowKeys);
      setselectedRows(selectedRows);
    }
  };

  const confirmAction = async () => {
    setloading(true);
    if (checkedStatus(selectedRows)) {
      const data = await confirmSelfBill(selectedRowKeys);
      if (data.code === '0000') {
        setselectedRowKeys([]);
        setselectedRows([]);
        reload();
        notification.success({ message: '确认成功' });
      }
      setloading(false);
    } else {
      setloading(false);
    }
  };

  const onDownload = async () => {
    if (selectedRowKeys.length && selectedRowKeys.length < 32) {
      setloading(true);
      const data = await downloadSelfBillDetail(selectedRowKeys);
      const blob = new Blob([data]);
      saveAs(blob, `账单明细-${format(new Date(), 'yyyyMMddHHmmss')}.xlsx`);
      setloading(false);
      setselectedRowKeys([]);
      setselectedRows([]);
      reload();
    } else {
      notification.error({ message: '最少下载1条数据账单或最多下载31条数据账单' });
      setloading(false);
    }
  };

  const gather = {
    onDownload,
    confirmAction,
  };

  const onToolBarRender = toolBarRender(gather);

  const rowSelection = {
    selectedRowKeys,
    onChange: onSelectChange,
  };

  const searchRender = ({ searchText, resetText }, { form }) => [
    <Button
      key="search"
      type="primary"
      style={{ marginRight: '10px' }}
      onClick={() => {
        // eslint-disable-next-line no-unused-expressions
        form?.submit();
      }}
    >
      {searchText}
    </Button>,
    <Button
      key="rest"
      style={{ marginRight: '10px' }}
      onClick={() => {
        // eslint-disable-next-line no-unused-expressions
        form?.resetFields();
      }}
    >
      {resetText}
    </Button>,
  ];
  return (
    <Spin tip="Loading..." spinning={loading} delay={100}>
      <PageHeaderWrapper>
        <ProTable
          actionRef={actionRef}
          request={params => query({ ...params })}
          columns={columns}
          rowKey={r => r.id}
          search={{
            collapsed: false,
            optionRender: searchRender,
          }}
          rowSelection={rowSelection}
          bordered
          toolBarRender={onToolBarRender}
          scroll={{ x: '100%' }}
        />
      </PageHeaderWrapper>
    </Spin>
  );
};

export default ReconciliationQuery;
