import React, { useState, useEffect, useContext, forwardRef, useImperativeHandle } from 'react';
import { Form, Button } from 'antd';
import { ServiceContext } from '../context';
import { TaskList, formItemLayout } from '../config';
import UploadImage from './UploadImage';
import commonStyle from '../common.less';

const FormRuleVPictures = forwardRef((props, ref) => {
  const { editData, specKeyItem } = props;
  const [imageList, setImageList] = useState({});
  const [commonImageList, setCommonImageList] = useState([]);
  const [detailImageList, setDetailImageList] = useState([]);
  const [cardImageList, setCardImageList] = useState([]);

  const [form] = Form.useForm();

  const customer = useContext(ServiceContext);
  useEffect(() => {
    if (customer.isEdit) {
      if (editData) {
        setImageList(editData.imageList);
        setCardImageList(editData.cardImageList);
        setCommonImageList(editData.commonImageList); // 编辑状态下设置公共图
        setDetailImageList(editData.detailImageList); // 编辑状态下设置详情图
        const editParams = {
          commonImageList: editData.commonImageList,
          detailImageList: editData.detailImageList,
        };
        if (customer.isCard) {
          editParams.cardImageList = editData.cardImageList;
        } else {
          editParams.imageList = editData.imageList;
        }
        const timer = setTimeout(() => {
          form.setFieldsValue(editParams);
          clearTimeout(timer);
        });
      }
    }
  }, [customer.isEdit, editData]);

  useEffect(() => {
    if (customer.isCard) return;
    if (specKeyItem.length) {
      const newImageList = specKeyItem.reduce((origin, item) => {
        const showItem = imageList[item] || [];
        origin[item] = showItem;
        return origin;
      }, {});
      setImageList(newImageList);
    }
  }, [specKeyItem]);

  const onCheck = async () => {
    try {
      const values = await form.validateFields();
      values.temp = 'infoImageData';
      return values;
    } catch (errorInfo) {
      console.log(errorInfo);
      return null;
    }
  };

  useImperativeHandle(ref, () => ({
    onCheck,
    setFieldsValue: form.setFieldsValue,
    getFieldsValue: form.getFieldsValue,
    reset: () => {
      form.resetFields();
      setImageList({});
      setCardImageList([]);
      setCommonImageList([]);
      setDetailImageList([]);
    },
  }));

  const onPictureSuccessEvent = (imgList, key) => {
    const newImgList = { ...imageList, [key]: imgList };
    setImageList(newImgList);
    form.setFieldsValue({
      imageList: newImgList,
    });
  };

  const onCommonSuccessEvent = imgList => {
    setCommonImageList(imgList);
    form.setFieldsValue({
      commonImageList: imgList,
    });
  };

  const onCardSuccessImageList = imgList => {
    setCardImageList(imgList);
    form.setFieldsValue({
      cardImageList: imgList,
    });
  };

  const onDetailSuccessImageList = imgList => {
    setDetailImageList(imgList);
    form.setFieldsValue({
      detailImageList: imgList,
    });
  };

  const [{ imgConfig }] = TaskList.filter(item => item.type === customer.productType);
  return (
    <Form
      form={form}
      {...formItemLayout}
      initialValues={{
        commonImageList: [],
        cardImageList: [],
      }}
    >
      <Form.Item
        name="commonImageList"
        label={imgConfig.commonImageList.title}
        extra={imgConfig.commonImageList.renderExtra(commonImageList.length)}
        rules={[
          {
            required: imgConfig.commonImageList.rule,
            type: 'array',
            message: `请上传${imgConfig.commonImageList.title}！`,
          },
          {
            validator(rule, value, callback) {
              if (customer.productType !== 1) {
                return callback();
              }
              const checkImageList = form.getFieldValue('imageList');
              const check = Object.keys(checkImageList).length;
              return check > 0 ? callback() : callback('请上传封面图片');
            },
          },
        ]}
      >
        <UploadImage
          disabled={customer.isService}
          name="commonImageList"
          limit={imgConfig.commonImageList.limit}
          pictures={commonImageList}
          setPictureList={list => onCommonSuccessEvent(list)}
        />
      </Form.Item>
      {customer.isCard && (
        <Form.Item
          name="cardImageList"
          label={imgConfig.cardImageList.title}
          extra={imgConfig.cardImageList.renderExtra(cardImageList.length)}
          rules={[
            {
              required: imgConfig.cardImageList.rule,
              type: 'array',
              message: `请上传${imgConfig.cardImageList.title}！`,
            },
          ]}
        >
          <UploadImage
            disabled={customer.isService}
            name="cardImageList"
            limit={imgConfig.cardImageList.limit}
            pictures={cardImageList}
            setPictureList={list => onCardSuccessImageList(list)}
          />
        </Form.Item>
      )}
      {!customer.isCard &&
        Object.keys(imageList).map(key => (
          <Form.Item
            name={['imageList', key]}
            key={key}
            label={`商品图片(${key})`}
            rules={[{ required: imgConfig.imageList.rule, type: 'array', message: '请上传图片！' }]}
            extra={imgConfig.imageList.renderExtra()}
          >
            <UploadImage
              disabled={customer.isService}
              name={key}
              limit={11}
              pictures={imageList[key]}
              setPictureList={list => onPictureSuccessEvent(list, key)}
            />
            <Button className={commonStyle.pullImage} type="primary">
              拉取公共图
            </Button>
          </Form.Item>
        ))}
      <Form.Item
        name="detailImageList"
        label={imgConfig.detailImageList.title}
        rules={[
          {
            type: 'array',
            required: imgConfig.detailImageList.rule,
            message: `请上传${imgConfig.detailImageList.title}！`,
          },
        ]}
        extra={imgConfig.detailImageList.renderExtra()}
      >
        <UploadImage
          disabled={customer.isService}
          limit={imgConfig.detailImageList.limit}
          name="detailImageList"
          pictures={detailImageList}
          setPictureList={list => onDetailSuccessImageList(list)}
        />
      </Form.Item>
    </Form>
  );
});

export default FormRuleVPictures;
