import React from 'react';
import { Button, Popover } from 'antd';
import UploadImage from '@/components/UploadImg/index.jsx';

// 品牌审核状态
/**
 * @description: 品牌审核状态 1-待审核
 */
export const auditStatusWait = 1;
/**
 * @description: 品牌审核状态 2-审核通过
 */
export const auditStatusPass = 2;
/**
 * @description: 品牌审核状态 3-审核拒绝
 */
export const auditStatusReject = 3;

/**
 * @description: 品牌审核状态枚举 0-无 1-待审核，2-审核通过，3-审核拒绝
 */
export const brandStatusEnum = {
  // 0: '-',
  [auditStatusWait]: $t('pending.review'),
  [auditStatusPass]: $t('review.passed'),
  [auditStatusReject]: $t('reject'),
};
/**
 * @description: 品牌审核 2审核通过 3驳回
 */
export const brandAuditEnum = {
  [auditStatusPass]: $t('review.passed'),
  [auditStatusReject]: $t('reject'),
};
// 操作状态 查看、修改、添加、补充资质
/**
 * @description: 品牌操作状态 查看
 */
export const brandActionFind = 'find';
/**
 * @description: 品牌操作状态 修改
 */
export const brandActionEdit = 'edit';
/**
 * @description: 品牌操作状态 添加
 */
export const brandActionAdd = 'add';
/**
 * @description: 品牌操作状态 补充资质
 */
export const brandActionSupplement = 'supplement';

/**
 * @description: 列表基础字段
 */
export const brandBaseColumn = [
  {
    title: $t('brand.name'),
    dataIndex: 'name',
    key: 'name',
    align: 'center',
  },
  {
    title: $t('chinese.name'),
    dataIndex: 'chineseName',
    key: 'chineseName',
    hideInSearch: true,
    align: 'center',
  },
  {
    title: $t('english.name'),
    key: 'englishName',
    dataIndex: 'englishName',
    hideInSearch: true,
    align: 'center',
  },
  {
    title: $t('brand.abbreviationalias'),
    key: 'alias',
    dataIndex: 'alias',
    hideInSearch: true,
    align: 'center',
  },
];

// 品牌列表字段
export const brandColumn = config => {
  const { onAction } = config;
  return [
    ...brandBaseColumn,
    {
      title: $t('review.status'),
      key: 'status',
      dataIndex: 'status',
      align: 'center',
      valueEnum: brandStatusEnum,
      render: (_, r) => {
        const { status } = r;
        return status === auditStatusReject ? (
          <Popover content={r.rejectReason} title={$t('rejection.reason')} trigger="hover">
            <Button type="link" danger>{$t('reject')}</Button>
          </Popover>
        ) : (
          brandStatusEnum[status]
        );
      },
    },
    {
      title: $t('actions'),
      hideInSearch: true,
      dataIndex: 'option',
      align: 'center',
      width: 200,
      render: (_, r) => [
        (r.modifiable && (
          <Button key="check" type="primary" onClick={() => onAction(r, 'edit')}>{$t('edit')}</Button>
        )) ||
          '',
        ([null, auditStatusReject].includes(r.status) && !r.modifiable && (
          <Button key="supplement" type="primary" ghost onClick={() => onAction(r, 'supplement')}>{$t('supplement.credentials')}</Button>
        )) ||
          '',
      ],
    },
  ];
};

// 品牌信息字段
export const brandInfoColumn = config => {
  const { actionStatus } = config;
  const disabled = brandActionSupplement === actionStatus;
  const baseInfo = [
    {
      title: $t('certificate'),
      dataIndex: 'qualifyUrl',
      formItemProps: {
        rules: [{ required: true, message: $t('select.certificate') }],
      },
      renderFormItem: () => (
        <UploadImage limit={30} accept={['jpg', 'jpeg', 'png', 'zip', 'rar', 'pdf']} />
      ),
    },
    {
      title: $t('authorization.certificate'),
      dataIndex: 'authorizationUrl',
      formItemProps: {
        rules: [{ required: true, message: $t('select.license') }],
      },
      renderFormItem: () => (
        <UploadImage limit={30} accept={['jpg', 'jpeg', 'png', 'zip', 'rar', 'pdf']} />
      ),
    },
    {
      title: $t('brand.name'),
      dataIndex: 'name',
      maxLength: 50,
      fieldProps: {
        disabled,
      },
      formItemProps: {
        rules: disabled ? [] : [{ required: true, message: $t('enter.brand.name') }],
      },
    },
    {
      title: $t('chinese.name'),
      dataIndex: 'chineseName',
      maxLength: 50,
      fieldProps: {
        disabled,
      },
    },
    {
      title: $t('english.name'),
      dataIndex: 'englishName',
      maxLength: 100,
      fieldProps: {
        disabled,
      },
    },
    {
      title: $t('brand.abbreviationalias'),
      dataIndex: 'alias',
      maxLength: 50,
      formItemProps: {
        rules: disabled ? [] : [{ required: true, message: $t('enter.brand.abbreviation') }],
      },
      fieldProps: {
        disabled,
      },
    },
    {
      title: $t('upload.rectangle.logo'),
      dataIndex: 'horizontalLogo',
      formItemProps: {
        rules: disabled ? [] : [{ required: true, message: $t('upload.rectangle.logo') }],
        extra: (
          <div>
            <div>{$t('size.21972')}</div>
            <div>{$t('guidelines.1.transparent.bg.2.2px.min.margin')}</div>
          </div>
        ),
      },
      fieldProps: {
        disabled,
      },
      renderFormItem: () => <UploadImage limit={1} disabled={disabled} width={219} height={72} />,
    },
    {
      title: $t('logo.upload'),
      dataIndex: 'logo',
      formItemProps: {
        rules: disabled ? [] : [{ required: true, message: $t('upload.logo') }],
        extra: <span>{$t('size.192192')}</span>,
      },
      fieldProps: {
        disabled,
      },
      renderFormItem: () => <UploadImage limit={1} disabled={disabled} width={192} height={192} />,
    },
  ];
  return baseInfo;
};
