import React, { useRef, useState } from 'react';
import { Form, Input, Button, Checkbox, Card, notification } from 'antd';
import { history } from 'umi';
import { da } from 'date-fns/locale';
import styles from './style.less';
import { passwordRule, resetPassword } from './service';

const Password = props => {
  const formRef = useRef();
  const [loading, setLoading] = useState(false);
  const goHome = () => {
    formRef.current.resetFields();
    history.push('/');
  };
  const submit = async record => {
    const data = await resetPassword({
      password: record.password,
      newPassword: record.newPassword,
    });
    if (data.code === 2000) {
      notification.success({
        message: '修改成功',
      });
      setTimeout(() => {
        goHome();
      }, 1000);
      setLoading(false);
    } else {
      setLoading(false);
      notification.error({
        message: data.detail || data.msg,
      });
    }
  };
  const rule = async record => {
    const data = await passwordRule({ password: record.newPassword });
    if (data.code === 2000) {
      submit(record);
    } else {
      setLoading(false);
      notification.warning({
        message: data.detail || data.msg,
      });
    }
  };
  const onFinish = values => {
    setLoading(true);
    rule(values);
  };
  return (
    <div className={styles.div}>
      <Card className={styles.card}>
        <p>修改密码</p>
        <Form
          initialValues={{ remember: true }}
          onFinish={onFinish}
          autoComplete="off"
          className={styles.form}
          ref={formRef}
        >
          <Form.Item name="password" rules={[{ required: true, message: '请输入原密码!' }]}>
            <Input.Password placeholder="输入原密码" />
          </Form.Item>
          <Form.Item
            name="newPassword"
            rules={[
              { required: true, message: '请输入新密码！' },
              {
                message: '至少13最多18位且必须包含大小写字母、数字和特殊字符',
                pattern: /(?=.*\d)(?=.*[a-z])(?=.*[A-Z])(?=.*[^a-zA-Z0-9]).{13,18}/,
              },
            ]}
          >
            <Input.Password placeholder="输入新密码" maxLength={18} />
          </Form.Item>
          <Form.Item
            name="newPasswordAgain"
            rules={[
              { required: true, message: '请输入新密码！' },
              {
                message: '至少13位最多18位且必须包含大小写字母、数字和特殊字符',
                pattern: /(?=.*\d)(?=.*[a-z])(?=.*[A-Z])(?=.*[^a-zA-Z0-9]).{13,18}/,
              },
              ({ getFieldValue }) => ({
                validator(val, value) {
                  if (value && getFieldValue('newPassword') !== value) {
                    // eslint-disable-next-line prefer-promise-reject-errors
                    return Promise.reject('两次密码输入不一致');
                  }
                  return Promise.resolve();
                },
              }),
            ]}
          >
            <Input.Password placeholder="输入新密码" maxLength={18} />
          </Form.Item>
          <Form.Item className={styles.btn}>
            <Button type="primary" htmlType="submit" loading={loading}>
              {' '}
              确认修改
            </Button>
            <Button type="primary" ghost onClick={goHome}>
              返回首页
            </Button>
          </Form.Item>
        </Form>
      </Card>
    </div>
  );
};

export default Password;
