import React, { useState, useEffect, forwardRef, useImperativeHandle } from 'react';
import { Cascader, Form, Input, Select } from 'antd';
import { useParams } from 'react-router-dom';
import { DefaultOptionType } from 'antd/lib/select';

interface CategoryItem {
  id: number;
  level: number;
  name: string;
  parentId: number;
  children?: CategoryItem[];
}
interface BrandItem {
  id: number;
  name: string;
  chineseName?: string;
  chinesePinyin?: string;
  englishName?: string;
  firstLetter?: string;
  horizontalLogo?: string;
  logo?: string;
  priority?: string;
}

interface PropsType {
  editData: any;
  categoryList: Array<CategoryItem>;
  brandList: BrandItem[];
}

const categoryIdArrayList = [
  {
    value: 'zhejiang',
    label: 'Zhejiang',
    children: [
      {
        value: 'hangzhou',
        label: 'Hangzhou',
        children: [
          {
            value: 'xihu',
            label: 'West Lake',
          },
        ],
      },
    ],
  },
  {
    value: 'jiangsu',
    label: 'Jiangsu',
    children: [
      {
        value: 'nanjing',
        label: 'Nanjing',
        children: [
          {
            value: 'zhonghuamen',
            label: 'Zhong Hua Men',
          },
        ],
      },
    ],
  },
];

const formItemLayout = {
  labelCol: {
    sm: { span: 3 },
  },
  wrapperCol: {
    sm: { span: 16 },
  },
};

const CreateSelectOption = (optionList: BrandItem[]) =>
  optionList.map((brandItem: BrandItem) => (
    <Select.Option key={brandItem.id} value={brandItem.id}>
      {brandItem.name}
    </Select.Option>
  ));

const fileterBrandOptions = (
  input: string,
  options: { key: string; value: number; children: string },
) => options.children.includes(input);

const filterCategoryOptions = (inputValue: string, path: DefaultOptionType[]) =>
  path.some(option => option.name.toLowerCase().indexOf(inputValue.toLowerCase()) > -1);

const FormInformationBasic = forwardRef((props: PropsType, ref) => {
  const { editData, categoryList, brandList } = props;
  const [form] = Form.useForm();

  const onCheck = async () => {
    try {
      const values = await form.validateFields();
      return values;
    } catch (errorInfo) {
      return null;
    }
  };

  useImperativeHandle(ref, () => ({
    onCheck,
  }));
  const { id } = useParams<any>();
  useEffect(() => {
    if (+id !== 0) {
      if (!Object.keys(editData).length) return;
      const { serviceItem } = editData.skuList[0];
      form.setFieldsValue({
        brandId: editData.brandId,
        supplierId: editData.supplierId,
        name: editData.name,
        categoryId: [editData.firstCategoryId, editData.secondCategoryId, editData.thirdCategoryId],
        description: serviceItem.description,
      });
    }
  }, [id, editData]);

  const onFinish = (values: any) => {
    console.log('Received values of form: ', values);
  };

  return (
    <Form
      {...formItemLayout}
      form={form}
      name="register"
      onFinish={onFinish}
      initialValues={{
        brandId: null,
        supplierId: '',
        name: '',
        categoryId: [],
        description: '',
      }}
      scrollToFirstError
    >
      <Form.Item
        name="brandId"
        label="供应商名称"
        rules={[{ required: true, message: '请选择供应商名称!' }]}
      >
        <Select showSearch placeholder="请选择供应商名称" filterOption={fileterBrandOptions}>
          {CreateSelectOption(brandList)}
        </Select>
      </Form.Item>
      <Form.Item
        name="name"
        label="商品名称"
        rules={[{ required: true, message: '请输入商品名称!', whitespace: true }]}
      >
        <Input placeholder="请输入商品名称" />
      </Form.Item>
      <Form.Item
        name="categoryId"
        label="商品类目"
        rules={[{ type: 'array', required: true, message: '请输入商品类目!' }]}
      >
        <Cascader
          placeholder="请选择商品类目！"
          showSearch={{ filter: filterCategoryOptions }}
          fieldNames={{ label: 'name', value: 'id', children: 'children' }}
          options={categoryList}
        />
      </Form.Item>

      <Form.Item
        name="description"
        label="描述"
        rules={[{ required: true, message: '请输入描述！' }]}
      >
        <Input.TextArea showCount maxLength={100} placeholder="请输入描述！" />
      </Form.Item>
    </Form>
  );
});

export default FormInformationBasic;
