package com.quantgroup.eos.configuration;

import java.util.ArrayList;
import java.util.Collections;
import org.springframework.boot.context.config.ConfigData;
import org.springframework.boot.context.config.ConfigDataLoader;
import org.springframework.boot.context.config.ConfigDataLoaderContext;
import org.springframework.boot.logging.DeferredLog;
import org.springframework.core.env.MapPropertySource;
import org.springframework.core.log.LogMessage;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

/**
 * 测试环境 通过Spring Boot Config Data Loader 模式来加载配置 测试环境 mysql、redis 等服务地址及端口号
 * 获取 eos 配置信息 API :
 * 如：<a href="http://eos.quantgroups.com/api/apollo/env_vars?namespace=yxm">...</a>
 * 此类配置信息加载完成后，交由 spring 管理
 */
public class EosConfigDataLoader implements ConfigDataLoader<EosConfigDataResource> {

    private static final String NAMESPACE = "NAMESPACE";
    private static final String ENV = "env";
    private static final String ENV_DEV = "dev";
    private static final String EOS_SERVER_HOST = "EOS_SERVER_HOST";

    @Override
    public ConfigData load(ConfigDataLoaderContext context, EosConfigDataResource resource) {
        DeferredLog logger = context.getBootstrapContext().get(DeferredLog.class);

        String env = System.getProperty(ENV);
        String namespace = System.getProperty(NAMESPACE);
        logger.info(LogMessage.of(() -> "Loading configuration from eos_server , env : " + env + " , NAMESPACE : " + namespace ));
        if(!ENV_DEV.equals(env)) {
            logger.info(LogMessage.of(() -> "Wow, the production environment. The configuration center is quiet. I dare not do anything." ));
            return new ConfigData(new ArrayList<>());
        }

        // When the namespace is not empty, the configuration can be loaded from the eos server.
        if (namespace == null || namespace.trim().isEmpty()) {
            logger.info(LogMessage.of(() -> "You don't seem to have configured NAMESPACE, right? Aren't you going to connect to the internal kubernetes?" ));
            return new ConfigData(new ArrayList<>());
        }

        try {
            RestTemplate restTemplate = new RestTemplate();
            String kubernetesServer = System.getProperty(EOS_SERVER_HOST, "http://eos.quantgroups.com/");
            String url = kubernetesServer + "api/apollo/env_vars?namespace=" + namespace;
            ResponseEntity<EosConfigInfo> response = restTemplate.getForEntity(url, EosConfigInfo.class);
            EosConfigInfo envInfo = response.getBody();
            if (envInfo != null && envInfo.isSuccess() && envInfo.getDetails() != null) {
                logger.info(LogMessage.of(() -> "The environment variables of kubernets have been init successed, and you can rest assured to start your service outside of kubernetes." ));
                return new ConfigData(
                        Collections.singletonList(new MapPropertySource("eosConfig", envInfo.getDetails())));
            }
        } catch (Exception e) {
            logger.error(LogMessage.of(() -> "The environment variable injection of kubernets failed, please troubleshoot."),e);
            return new ConfigData(new ArrayList<>());
        }
        return new ConfigData(new ArrayList<>());
    }
}