const path = require('path')
const Client = require('kubernetes-client').Client
const config = require('kubernetes-client').config
const _ = require('lodash')
const moment = require('moment')
const yaml = require('js-yaml')
const logger = require('koa-log4').getLogger('kubeService')
const yamls = require('../yamls')
const dict = require('./dictService')

const client = new Client({
  config: config.fromKubeconfig(
    path.resolve(__dirname, './kubeConfig.yaml'),
  ),
  version: '1.10',
})

const makeManifest = (data) => {
  if (!data.debug) {
    data.debug = '"0"'
  }
  const yamlManifest = yamls[data.type].replace(/{{([A-Za-z0-9_\.]+)}}/g, function () {
    if (_.get(data, arguments[1], null) === null) {
      throw new Error(`缺少模板所需变量: ${arguments[1]}`)
    }
    return _.get(data, arguments[1])
  })

  const yamlArray = yamlManifest.split('---')
  return yamlArray
}

const serviceCreate = async (data) => {
  const { namespace, serviceName } = data
  const yamlArray = makeManifest(data)

  for (const item of yamlArray) {
    const jsonObj = yaml.load(item);

    let pvc
    let pvcName

    switch (jsonObj.kind) {
      case 'Service':
        await client.api.v1.namespaces(namespace).services.post({ body: jsonObj })
        break;

      case 'Deployment':
        logger.info('Deployment:', JSON.stringify(jsonObj))
        await client.apis.apps.v1beta1.namespaces(namespace).deployments.post({ body: jsonObj })
        break;

      case 'PersistentVolumeClaim':
        pvcName = `${serviceName}-${namespace}`
        pvc = await client.api.v1.namespaces(namespace).persistentvolumeclaims.get()
        pvc = pvc.body.items.filter(item => item.metadata.name === pvcName)
        if (!pvc.length) {
          await client.api.v1.namespaces(namespace).persistentvolumeclaims.post({ body: jsonObj })
        }
        break;

      default:
        break;
    }
  }
}

const serviceUpdate = async (data) => {
  const { namespace, serviceName } = data
  const yamlArray = makeManifest(data)

  for (const item of yamlArray) {
    const jsonObj = yaml.load(item);
    if (jsonObj.kind === 'Deployment') {
      logger.info('Deployment:', JSON.stringify(jsonObj))
      await client.apis.apps.v1beta1.namespaces(namespace).deployments(serviceName).put({ body: jsonObj })
    }
  }
}

const formatServiceInfo = (obj) => {
  const portObj = {}
  obj.spec.ports.forEach((i) => {
    portObj[`${i.name}_port`] = i.nodePort
  })

  return _.assign(portObj, {
    clusterIp: obj.spec.clusterIP,
    portMappings: obj.spec.ports,
    // labels: obj.metadata.labels,
  })
}

const getPodStatus = (podInfo) => {
  if (podInfo.status.phase === 'Pending') {
    return 'Pending'
  }
  if (podInfo.status.conditions[2].status === 'False') {
    return 'PodScheduling'
  }
  if (podInfo.status.conditions[0].status === 'False') {
    return 'Initializing'
  }
  if (podInfo.status.conditions[1].status === 'False') {
    return 'Waiting'
  }
  if (podInfo.metadata.deletionTimestamp) {
    return 'Terminating'
  }
  return 'Normal'
}

const formatPodInfo = (podInfo) => {
  // if (podInfo.metadata.name.indexOf('xyqb') !== -1) {
  //   console.log(1, JSON.stringify(podInfo), podInfo.metadata.labels['qcloud-app'])
  // }
  const podStatus = getPodStatus(podInfo)

  // const containerImage = _.get(podInfo.status.containerStatuses, '[0].image', '')
  const containerImage = _.get(podInfo.spec.containers, '[0].image', '')

  const ret = {
    serviceName: (podInfo.metadata.labels && podInfo.metadata.labels['qcloud-app']) || podInfo.metadata.name,
    podName: podInfo.metadata.name,
    status: podStatus,
    podIp: podInfo.status.podIP,
    lanIp: podInfo.status.hostIP,
    startTime: podInfo.status.startTime,
    createdAt: moment(new Date(podInfo.status.startTime)).startOf('minute').fromNow(),
    image: containerImage,
    labels: podInfo.metadata.labels,
  }

  if (containerImage !== '') {
    ret.branch = containerImage.split(':')[1]
    ret.branch_no_time = containerImage.split(':')[1].split('-')[0]
    ret.branch_time = containerImage.split(':')[1].split('-')[1]
  }

  return ret
}

const formatIngressInfo = obj => ({ host: _.get(obj.spec, 'rules[0].host', '') })

const getPods = async (namespace) => {
  const podData = await client.api.v1.namespaces(namespace).pods.get()
  return podData.body.items
}

const getServices = async (namespace) => {
  const ret = []
  const service = {}

  const res = await Promise.all([
    client.api.v1.namespaces(namespace).pods.get(),
    client.api.v1.namespaces(namespace).services.get(),
    client.apis.extensions.v1beta1.namespaces(namespace).ingresses.get(),
  ])

  res[0].body.items.forEach(async (item) => {
    const serviceName = (item.metadata.labels && item.metadata.labels['qcloud-app']) || item.metadata.name
    service[serviceName] = formatPodInfo(item)
  })


  res[1].body.items.forEach(async (item) => {
    if (service[item.metadata.name]) {
      service[item.metadata.name] = _.assign(service[item.metadata.name], formatServiceInfo(item))
    }
  })

  res[2].body.items.forEach(async (item) => {
    if (service[item.metadata.name]) {
      service[item.metadata.name] = _.assign(service[item.metadata.name], formatIngressInfo(item))
    }
  })

  for (const index in service) {
    if (Object.prototype.hasOwnProperty.call(service, index)) {
      ret.push(service[index])
    }
  }

  return ret
}

const getServiceDetail = async (namespace, name) => {
  const res = await Promise.all([
    client.api.v1.namespaces(namespace).pods.get({ qs: { labelSelector: `qcloud-app=${name},tier!=job` } }),
    client.api.v1.namespaces(namespace).services(name).get(),
  ])

  res[0] = formatPodInfo(res[0].body.items[0])
  res[1] = formatServiceInfo(res[1].body)

  if (!dict.commonService.includes(name)) {
    res[3] = await client.apis.extensions.v1beta1.namespaces(namespace).ingresses(name).get()
    res[3] = formatIngressInfo(res[3].body)
  }

  return _.assign({}, res[0], res[3], res[1])
}

const serviceRestart = async (namespace, name) => {
  await client.api.v1.namespaces(namespace).pods(name).delete()
}

const serviceDelete = async (namespace, name, rsName) => {
  await client.apis.apps.v1beta1.namespaces(namespace).deployments(name).delete()
  await client.apis.apps.v1.namespaces(namespace).replicasets(rsName).delete()
  await client.api.v1.namespaces(namespace).services(name).delete()
}

module.exports = {
  getServiceDetail,
  getServices,
  getPods,
  serviceCreate,
  serviceUpdate,
  serviceRestart,
  serviceDelete,
}
