const Router = require('koa-router')
const yaml = require('js-yaml')
const _ = require('lodash')
const logger = require('koa-log4').getLogger()

const templates = require('../serviceTemplate')
const lruCache = require('../services/lruCache.service')
const { ingressCreate, ingressDelete } = require('../kubeService/ingress')
const { podGet, serviceCreate } = require('../kubeService/service')

const router = new Router()
module.exports = router

router.get('/', async (ctx) => {
  // 取节点列表的第一个作为服务的访问ip
  const cacheKey = 'k8s.nodes.first'
  const c = lruCache.get(cacheKey)
  let lanIp = lruCache.get(cacheKey)
  if (!c) {
    const res = await ctx.cluster.node_list()
    lanIp = _.get(res, 'nodes[0].lanIp', '')
    lruCache.set(cacheKey, lanIp)
  }

  const data = await ctx.cluster.service_list(ctx.query.namespace)
  const podData = await podGet(ctx.query.namespace)

  const getDetail = async (item) => {
    const resData = await ctx.cluster.service_get(item.serviceName, ctx.query.namespace)

    item.image = resData.service.containers[0].image
    if (item.userLabels.type === 'base') {
      item.portMappings = resData.service.portMappings
      const pod = podData.body.items.filter(i => i.metadata.name.indexOf(item.serviceName) !== -1)
      lanIp = pod[0].status.hostIP
      item.lanIp = lanIp
    }
  }
  const task = []
  for (let i = 0; i < data.services.length; i += 1) {
    task.push(getDetail(data.services[i]))
  }
  await Promise.all(task)
  ctx.body = ctx.ok(data)
})

const createService = async (ctx) => {
  const {
    type, serviceName, namespace, image, system_name, domain, label,
  } = ctx.request.body
  logger.info('创建服务', ctx.request.body)
  if (serviceName === 'mysql') {
    await serviceCreate(namespace, serviceName, image, label)
    ctx.body = ctx.ok('创建成功')
    return
  }
  const data = {
    serviceName,
    namespace,
    image,
    system_name,
  }
  if (!system_name) {
    // ui abTest的时候不一样
    data.system_name = serviceName
  }

  // java 增加内存限制
  if (type === 'java') {
    data.resources = {
      cpu: {
        request: 100,
        limit: 250,
      },
      memory: {
        request: 100,
        limit: 1024,
      },
    }

    const special = ['xyqb', 'xyqb-user2']
    if (special.includes(serviceName)) {
      data.resources.memory.request = 600
    }
  }

  const template = templates[type].replace(/{{([A-Za-z0-9_\.]+)}}/g, function () {
    console.log(arguments[1])
    if (_.get(data, arguments[1], null) === null) {
      throw new Error(`缺少模板所需变量: ${arguments[1]}`)
    }
    return _.get(data, arguments[1])
  })
  const params = yaml.load(template)
  await ctx.cluster.service_create(params, label)
  if (label !== 'base') {
    if (serviceName === 'xyqb-user2') {
      await ingressCreate(namespace, 'xyqb-user2-2', 'passportapi2')
    }
    await ingressCreate(namespace, serviceName, domain)
  }
  ctx.body = ctx.ok('创建成功')
}

router.post('/create', async (ctx) => {
  await createService(ctx)
})

router.post('/details', async (ctx) => {
  ctx.validate(ctx.Joi.object().keys({
    serviceName: ctx.Joi.string().required(),
    namespace: ctx.Joi.string().required(),
  }))

  const res = await ctx.cluster.node_list()
  const lanIp = _.get(res, 'nodes[0].lanIp', '')

  const data = await ctx.cluster.service_get(ctx.request.body.serviceName, ctx.request.body.namespace)
  ctx.body = ctx.ok(Object.assign({}, data, { lanIp }))
})
router.post('/delete', async (ctx) => {
  await ctx.cluster.service_delete(ctx.request.body.serviceName, ctx.request.body.namespace)
  await ingressDelete(ctx.request.body.namespace, ctx.request.body.serviceName)
  ctx.body = ctx.ok('删除成功')
})

router.post('/modifyImage', async (ctx) => {
  let list = await ctx.cluster.service_list(ctx.request.body.namespace)
  list = list.services.map(item => item.serviceName)

  if (list.includes(ctx.request.body.serviceName)) {
    await ctx.cluster.service_modifyImage(ctx.request.body.serviceName, ctx.request.body.image, ctx.request.body.namespace)
  } else {
    await createService(ctx)
  }

  ctx.body = ctx.ok('更新成功')
})
router.post('/instance', async (ctx) => {
  const data = await ctx.cluster.instance_get(ctx.request.body.serviceName, ctx.request.body.namespace)
  ctx.body = ctx.ok(data)
})
router.post('/redeploy', async (ctx) => {
  await ctx.cluster.service_redeployment(ctx.request.body.serviceName, ctx.request.body.namespace)
  ctx.body = ctx.ok('服务重部署成功')
})
