const _ = require('lodash')
const moment = require('moment')
const yaml = require('js-yaml')
const logger = require('koa-log4')
  .getLogger('kubeServiceHandle')
const yamls = require('../yamls')
const APP_CONFIG = require('../config')

const keepNamespace = ['default', 'kube-system', 'monitor', 'kube-public']
const getAllNamespace = async (client) => {
  const res = await client.api.v1.namespace.get()
  const data = {
    namespaces: [],
  }
  res.body.items.forEach((item) => {
    if (!keepNamespace.includes(item.metadata.name)) {
      data.namespaces.push({
        name: item.metadata.name,
        description: item.metadata.annotations ? item.metadata.annotations.description : '',
        owner: item.metadata.annotations ? item.metadata.annotations.owner : '',
        status: item.status ? item.status.phase : '',
        createdAt: moment(item.metadata.creationTimestamp)
          .format('YYYY-MM-DD HH:mm:ss'),
      })
    }
  })
  return data
}

const makeManifest = (data) => {
  if (!data.debug) {
    data.debug = '"0"'
  }

  if (!data.mock) {
    data.mock = '"0"'
  }

  if (data.label === 'java') {
    data.serviceType = data.debug === '"0"' ? 'ClusterIP' : 'NodePort'
  }

  const yamlManifest = yamls[data.type].replace(/{{([A-Za-z0-9_\.]+)}}/g, function () {
    if (_.get(data, arguments[1], null) === null) {
      throw new Error(`缺少模板所需变量: ${arguments[1]}`)
    }
    return _.get(data, arguments[1])
  })

  const manifestArray = yamlManifest.split('---')
  return manifestArray
}

const createNamespace = async (client, name, description) => {
  const parmas = {
    apiVersion: 'v1',
    kind: 'Namespace',
    metadata: {
      annotations: {
        description,
      },
      name,
    },
    spec: {
      imagePullSecrets: [
        {
          name: 'qcloudregistrykey',
        },
        {
          name: 'tencenthubkey',
        },
      ],
    },
  }
  logger.info('创建namespace', name, JSON.stringify(parmas))
  await client.api.v1.namespaces.post({ body: parmas })

  const manifestArray = makeManifest({ namespace: name, type: 'tencenthubkey' })
  for (const item of manifestArray) {
    const manifest = yaml.load(item);
    logger.info('创建secret', JSON.stringify(manifest))
    await client.api.v1.namespaces(name)
      .secrets
      .post({ body: manifest })
  }
}


const serviceCreate = async (cluterParams, client, data) => {
  data.cluster = cluterParams.cluster
  data.hosts = cluterParams.hosts
  const { namespace, serviceName } = data
  const manifestArray = makeManifest(data)

  for (const item of manifestArray) {
    let manifest = yaml.load(item);
    let pvc
    let pvcName

    switch (manifest.kind) {
      case 'Service':
        if (data.wechat === '"1"') {
          manifest.spec.type = 'NodePort'
        }
        try {
          logger.info('创建svc', JSON.stringify(manifest))
          await client.api.v1.namespaces(namespace)
            .services
            .post({ body: manifest })
        } catch (error) {
          logger.warn(error.toString())
        }
        break;

      case 'Deployment':
        if (APP_CONFIG.noHealthCheckApp.includes(serviceName)) {
          manifest = _.omit(manifest, ['spec.template.spec.containers[0].readinessProbe'])
          manifest = _.omit(manifest, ['spec.template.spec.containers[0].livenessProbe'])
        }
        logger.info('创建deploy', serviceName, JSON.stringify(manifest))

        await client.apis.apps.v1beta1.namespaces(namespace)
          .deployments
          .post({ body: manifest })
        break;

      case 'PersistentVolumeClaim':
        pvcName = `${serviceName}-${namespace}`
        pvc = await client.api.v1.namespaces(namespace)
          .persistentvolumeclaims
          .get()
        pvc = pvc.body.items.filter(item => item.metadata.name === pvcName)
        if (!pvc.length) {
          logger.info('创建pvc', JSON.stringify(manifest))
          await client.api.v1.namespaces(namespace)
            .persistentvolumeclaims
            .post({ body: manifest })
        }
        break;

      default:
        break;
    }
  }
}

const imageUpdate = async (client, data) => {
  const { namespace, serviceName } = data
  const image = `ccr.ccs.tencentyun.com/${data.image}`
  const updateObj = {
    spec: {
      template: {
        spec: {
          containers: [{
            name: serviceName,
            image,
          }],
        },
      },
    },
  }

  logger.info('更新镜像', namespace, JSON.stringify(updateObj))
  await client.apis.apps.v1beta1.namespaces(namespace)
    .deployments(serviceName)
    .patch({ body: updateObj })
}

const deployUpdate = async (cluterParams, client, data) => {
  data.cluster = cluterParams.cluster
  data.hosts = cluterParams.hosts
  const { namespace, serviceName } = data
  const manifestArray = makeManifest(data)

  for (const item of manifestArray) {
    let manifest = yaml.load(item);
    if (APP_CONFIG.noHealthCheckApp.includes(serviceName)) {
      manifest = _.omit(manifest, ['spec.template.spec.containers[0].readinessProbe'])
      manifest = _.omit(manifest, ['spec.template.spec.containers[0].livenessProbe'])
    }
    if (manifest.kind === 'Deployment') {
      logger.info('更新deploy:', JSON.stringify(manifest))
      await client.apis.apps.v1beta1.namespaces(namespace)
        .deployments(serviceName)
        .put({ body: manifest })
    }
  }
}

const formatServiceInfo = (obj) => {
  const portObj = {}
  if (obj.spec.type === 'NodePort') {
    obj.spec.ports.forEach((i) => {
      portObj[`port_${i.port}`] = i.nodePort
    })
  }

  return _.assign(portObj, {
    clusterIp: obj.spec.clusterIP,
    serviceType: obj.spec.type,
    portMappings: obj.spec.ports,
  })
}

const getPodStatus = (podInfo) => {
  if (podInfo.metadata.deletionTimestamp) {
    return 'Terminating'
  }
  let status
  if (!podInfo.status.containerStatuses) {
    status = 'Pending'
  } else {
    if (podInfo.status.containerStatuses[0].state.waiting) {
      status = 'ContainerCreating'
    }
    if (podInfo.status.containerStatuses[0].state.running && !podInfo.status.containerStatuses[0].ready) {
      status = 'Waiting'
    }
    if (podInfo.status.containerStatuses[0].state.running && podInfo.status.containerStatuses[0].ready) {
      status = 'Normal'
    }
  }
  return status
}

const formatPodInfo = (podInfo) => {
  const podStatus = getPodStatus(podInfo)
  // for debug
  // if (podInfo.metadata.name.indexOf('zookeeper') !== -1) {
  //   console.log(1, podStatus, JSON.stringify(podInfo))
  // }
  const imageID = _.get(podInfo.status.containerStatuses, '[0].imageID', '')
  const image = _.get(podInfo.spec.containers, '[0].image', '')

  const ret = {
    serviceName: (podInfo.metadata.labels && podInfo.metadata.labels['qcloud-app']) || podInfo.metadata.name,
    podName: podInfo.metadata.name,
    status: podStatus,
    podIp: podInfo.status.podIP,
    lanIp: podInfo.status.hostIP,
    startTime: podInfo.status.startTime,
    createdAt: moment(new Date(podInfo.status.startTime))
      .startOf('minute')
      .fromNow(),
    image,
    imageID,
    labels: podInfo.metadata.labels,
  }

  // if (image !== '') {
  //   ret.branch = image.split(':')[1]
  // }

  return ret
}

const formatIngressInfo = obj => ({ host: _.get(obj.spec, 'rules[0].host', '') })

const getPods = async (client, namespace) => {
  const data = await client.api.v1.namespaces(namespace)
    .pods
    .get()
  return data
}

const podGetstatus = async client => client.api.v1.pods.get()

const getServicesFormat = async (client, namespace) => {
  const ret = []
  const service = {}

  const res = await Promise.all([
    client.api.v1.namespaces(namespace)
      .pods
      .get(),
    client.api.v1.namespaces(namespace)
      .services
      .get(),
    client.apis.extensions.v1beta1.namespaces(namespace)
      .ingresses
      .get(),
  ])

  res[0].body.items.forEach(async (item) => {
    const serviceName = (item.metadata.labels && item.metadata.labels['qcloud-app']) || item.metadata.name
    if (item.status.phase !== 'Failed') {
      service[serviceName] = formatPodInfo(item)
    }
  })


  res[1].body.items.forEach(async (item) => {
    if (service[item.metadata.name]) {
      service[item.metadata.name] = _.assign(service[item.metadata.name], formatServiceInfo(item))
    }
  })

  res[2].body.items.forEach(async (item) => {
    if (service[item.metadata.name]) {
      service[item.metadata.name] = _.assign(service[item.metadata.name], formatIngressInfo(item))
    }
  })

  for (const index in service) {
    if (Object.prototype.hasOwnProperty.call(service, index)) {
      ret.push(service[index])
    }
  }

  return ret
}

const getServiceDetail = async (client, namespace, name, type) => {
  const res = await Promise.all([
    client.api.v1.namespaces(namespace)
      .pods
      .get({ qs: { labelSelector: `qcloud-app=${name}` } }),
    client.api.v1.namespaces(namespace)
      .services(name)
      .get(),
  ])

  res[0] = formatPodInfo(res[0].body.items[0])
  res[1] = formatServiceInfo(res[1].body)

  // if (!dict.commonService.includes(name)) {
  if (type !== 'base') {
    res[2] = await client.apis.extensions.v1beta1.namespaces(namespace)
      .ingresses(name)
      .get()
    res[2] = formatIngressInfo(res[2].body)
  }

  return _.assign({}, res[0], res[1], res[2])
}

const serviceRestart = async (client, namespace, name) => {
  logger.info('重置服务', namespace, name)
  await client.api.v1.namespaces(namespace)
    .pods(name)
    .delete()
}

const serviceDelete = async (client, namespace, name) => {
  try {
    logger.info('删除deploy', namespace, name)
    await client.apis.apps.v1beta1.namespaces(namespace)
      .deployments(name)
      .delete()
    if (!~name.indexOf('mysql')) {
      logger.info('删除svc', namespace, name)
      await client.api.v1.namespaces(namespace)
        .services(name)
        .delete()
    }
  } catch (error) {
    logger.warn(error.toString())
  }
}

// const getReplicaSet = async (client, namespace) => {
//   const rsData = await client.apis.apps.v1beta2.namespaces(namespace)
//     .replicasets
//     .get()
//   return rsData.body.items
// }

const replicaSetDelete = async (client, namespace, name) => {
  logger.info('删除rs', namespace, name)
  await client.apis.apps.v1.namespaces(namespace)
    .replicasets
    .delete({ qs: { labelSelector: `qcloud-app=${name}` } })
  // await client.apis.apps.v1.namespaces(namespace).replicasets(rsName).delete()
}
const pvcDelete = async (client, namespace, name) => {
  try {
    if (!~name.indexOf('mysql')) {
      logger.info('删除pvc', namespace, name)
      await client.api.v1.namespaces(namespace)
        .persistentvolumeclaim(`${name}-${namespace}`)
        .delete()
    }
  } catch (error) {
    logger.warn(error.toString())
  }
}
const getServices = async (client, namespace) => {
  const data = await client.api.v1.namespaces(namespace)
    .services
    .get()
  return data
}

module.exports = {
  getAllNamespace,
  createNamespace,
  getServiceDetail,
  getServicesFormat,
  getServices,
  getPods,
  podGetstatus,
  serviceCreate,
  deployUpdate,
  serviceRestart,
  serviceDelete,
  imageUpdate,
  replicaSetDelete,
  pvcDelete,
}
