const request = require('request')
const crypto = require('crypto')
const domainConfig = require('../config')

class Client {
  constructor() {
    this.secretId = domainConfig.secretId
    this.secretKey = domainConfig.secretKey
    this.region = domainConfig.region
    this.clusterId = domainConfig.clusterId
    this.method = domainConfig.method
    this.protocal = domainConfig.protocal
    this.endpoint = ''
    this.path = domainConfig.path
    this.domain = domainConfig.domain

    // 禁止操作以下命名空间
    this.defaultNamespaces = ['kube-system', 'default']
  }

  sign(params, signMethod) {
    let str = ''
    const keys = Object.keys(params)
    keys.sort()
    for (const key of keys) {
      str += `&${key}=${params[key]}`
    }
    str = str.slice(1)
    const signStr = `${this.method + this.endpoint + this.path}?${str}`

    const signMethodMap = {
      HmacSHA1: 'sha1',
      HmacSHA256: 'sha256',
    }
    const hmac = crypto.createHmac(signMethodMap[signMethod], this.secretKey)
    return hmac.update(Buffer.from(signStr, 'utf-8')).digest('base64')
  }

  // 公共参数
  // Action: '',
  // Region: '',
  // Timestamp: null,
  // Nonce: null,
  // SecretId: '',
  // Signature: ''
  formatRequestData(action, params = {}) {
    params.Action = action
    params.Region = this.region
    params.Timestamp = Math.round(Date.now() / 1000)
    params.Nonce = Math.round(Math.random() * 65535)
    params.SecretId = this.secretId
    params.SignatureMethod = params.SignatureMethod || 'HmacSHA256'
    params.Signature = this.sign(params, params.SignatureMethod)
  }

  post(action, params = {}) {
    if (params.namespace && this.defaultNamespaces.includes(params.namespace)) {
      return Promise.reject(new Error('未授权的namespace'))
    }

    this.formatRequestData(action, params)
    const url = this.protocal + this.endpoint + this.path
    const self = this
    return new Promise(((resolve, reject) => {
      request({
        method: self.method,
        url,
        json: true,
        form: params,
      }, (err, res, body) => {
        if (err) {
          // console.error(err)
          reject(err)
        } else if (body.code === 0) {
          resolve(body.data)
        } else {
          reject(body)
        }
      })
    }))
  }
}

module.exports = Client
