const request = require('request')
const crypto = require('crypto')

class Client  {
  constructor ({secretId, secretKey, region, clusterId}) {
    this.secretId = secretId
    this.secretKey = secretKey
    this.region = region
    this.clusterId = clusterId
    this.method = 'POST'
    this.protocal = 'https://'
    this.endpoint = 'ccs.api.qcloud.com'
    this.path = '/v2/index.php'
    this.domain = ".lkbang.com"

    // 禁止操作以下命名空间
    this.defaultNamespaces = ['default', 'kube-system', 'kube-public']
  }

  sign (params, signMethod) {
    let str = ''
    let keys = Object.keys(params)
    keys.sort()
    for (let key of keys) {
      str += '&' + key + '=' + params[key]
    }
    str = str.slice(1)
    let signStr = this.method + this.endpoint + this.path + '?' + str

    let signMethodMap = {
        HmacSHA1: "sha1",
        HmacSHA256: "sha256"
    }
    let hmac = crypto.createHmac(signMethodMap[signMethod], this.secretKey)
    return hmac.update(Buffer.from(signStr, 'utf-8')).digest('base64')
  }

  // 公共参数
  // Action: '',
  // Region: '',
  // Timestamp: null,
  // Nonce: null,
  // SecretId: '',
  // Signature: ''
  formatRequestData (action, params = {}) {
    params.Action = action
    params.Region = this.region
    params.Timestamp = Math.round(Date.now() / 1000)
    params.Nonce = Math.round(Math.random() * 65535)
    params.SecretId = this.secretId
    params.SignatureMethod = params.SignatureMethod || 'HmacSHA256'
    params.Signature = this.sign(params, params.SignatureMethod)
  }

  post (action, params = {}) {
    if (params.namespace && this.defaultNamespaces.includes(params.namespace)) {
      return Promise.reject(new Error('未授权的namespace'))
    }

    this.formatRequestData(action, params)
    let url = this.protocal + this.endpoint + this.path
    let self = this
    return new Promise(function (resolve, reject) {
      request({
        method: self.method,
        url, 
        json: true,
        form: params
      }, function (err, res, body) {
        if (err) {
          console.error(err)
          reject(err)
        } else {
          if (body.code === 0) {
            resolve(body.data)
          } else {
            reject(body)
          }
        }
      })
    })
  }

  namespace_get () {
    return this.post('DescribeClusterNameSpaces', {clusterId: this.clusterId})
  }

  namespace_create (name) {
    return this.post('CreateClusterNamespace', {name, description: '', clusterId: this.clusterId})
  }

  namespace_delete (name) {
    return this.post('DeleteClusterNamespace', {"names.0": name, clusterId: this.clusterId})
  }

  service_list (namespace) {
    return this.post('DescribeClusterService', {namespace, clusterId: this.clusterId, allnamespace: 0})
  }

  service_get (serviceName, namespace) {
    return this.post('DescribeClusterServiceInfo', {serviceName, namespace, clusterId: this.clusterId})
  }

  async service_create (params) {
    params.clusterId = this.clusterId
    let str = JSON.stringify(params)
    await this.post('CreateClusterService', params)
    // 腾讯云api暂只能通过修改服务来改成Recreate策略
    let modifyParams = JSON.parse(str)
    modifyParams.strategy = 'Recreate'
    return this.post('ModifyClusterService', modifyParams)
  }

  service_delete (serviceName, namespace) {
    return this.post('DeleteClusterService', {serviceName, namespace, clusterId: this.clusterId})
  }

  service_modifyImage (serviceName, image, namespace) {
    // 修改示例的第一个容器
    return this.post('ModifyClusterServiceImage', {serviceName, namespace, image, clusterId: this.clusterId})
  }

  ingress_get (namespace) {
    return this.post('DescribeIngress', {namespace, clusterId: this.clusterId})
  }

  ingress_create (namespace) {
    return this.post('CreateIngress', {ingressName: `qa-${namespace}`, ingressDesc: '', namespace, clusterId: this.clusterId})
  }

  ingress_delete (ingressName, namespace) {
    return this.post('DeleteIngress', {ingressName, namespace, clusterId: this.clusterId})
  }

  ingress_modify (ingressName, namespace, rules) {
    return this.post('MosifyIngress', {ingressName, namespace, ...rules, clusterId: this.clusterId})
  }

}

exports.create = function (config) {
  return new Client(config)
}

// let client = new Client({
//   secretId: 'AKID4rY7wwNphsUcaFsy1pRtKhQvDj4CA3Ac',
//   secretKey: 'YBduRnjgVRGzmagZJbss3Vo58wWCyhgc',
//   clusterId: 'cls-acfx4pvj',
//   region: 'ap-beijing'
// })

// client.post('DescribeClusterNameSpaces', {clusterId: 'cls-acfx4pvj'}).then(res => {
//   console.log(5, res)
// })