const path = require('path')
const Client = require('kubernetes-client').Client
const config = require('kubernetes-client').config
const _ = require('lodash')
const yaml = require('js-yaml')
const logger = require('koa-log4').getLogger('kubeService')
const yamls = require('../yamls')

const client = new Client({
  config: config.fromKubeconfig(
    path.resolve(__dirname, './kubeConfig.yaml'),
  ),
  version: '1.10',
})

const serviceGet = async (namespace, service) => client.api.v1.namespaces(namespace).services(service).get()
const podGet = async namespace => client.api.v1.namespaces(namespace).pods.get()

const serviceCreate = async (data) => {
  const { type, namespace, serviceName } = data
  const yamlManifest = yamls[type].replace(/{{([A-Za-z0-9_\.]+)}}/g, function () {
    if (_.get(data, arguments[1], null) === null) {
      throw new Error(`缺少模板所需变量: ${arguments[1]}`)
    }
    return _.get(data, arguments[1])
  })

  const yamlArray = yamlManifest.split('---')

  for (const item of yamlArray) {
    const jsonObj = yaml.load(item);

    let pvc
    let pvcName

    switch (jsonObj.kind) {
      case 'Service':
        await client.api.v1.namespaces(namespace).services.post({ body: jsonObj })
        break;

      case 'Deployment':
        logger.info('Deployment:', JSON.stringify(jsonObj))
        await client.apis.apps.v1beta1.namespaces(namespace).deployments.post({ body: jsonObj })
        break;

      case 'PersistentVolumeClaim':
        pvcName = `${serviceName}-${namespace}`
        pvc = await client.api.v1.namespaces(namespace).persistentvolumeclaims.get()
        pvc = pvc.body.items.filter(item => item.metadata.name === pvcName)
        if (!pvc.length) {
          await client.api.v1.namespaces(namespace).persistentvolumeclaims.post({ body: jsonObj })
        }
        break;

      default:
        break;
    }
  }
}

const formatServiceInfo = (obj) => {
  const portObj = {}
  obj.spec.ports.forEach((i) => {
    portObj[`${i.name}_port`] = i.nodePort
  })

  return _.assign(portObj, {
    clusterIp: obj.spec.clusterIP,
    ports: obj.spec.ports,
    userLabels: obj.metadata.labels,
  })
}

const formatPodInfo = (podInfo) => {
  const podStatus = podInfo.status.phase === 'Running' && podInfo.status.conditions.every(i => i.status === 'True')
    ? 'Running' : 'Pending'

  const containerImage = _.get(podInfo.status.containerStatuses, '[0].image', '')

  const ret = {
    serviceName: (podInfo.metadata.labels && podInfo.metadata.labels['qcloud-app']) || podInfo.metadata.name,
    type: podInfo.metadata.labels.tier,
    pod_name: podInfo.metadata.name,
    pod_status: podStatus,
    pod_ip: podInfo.status.podIP,
    host_ip: podInfo.status.hostIP,
    start_time: podInfo.status.startTime,
    // live: moment(new Date(podInfo.status.startTime)).startOf('minute').fromNow(),
    image: containerImage,
  }

  if (containerImage !== '') {
    ret.branch = containerImage.split(':')[1]
    ret.branch_no_time = containerImage.split(':')[1].split('-')[0]
    ret.branch_time = containerImage.split(':')[1].split('-')[1]
  }

  return ret
}

const formatIngressInfo = obj => ({ host: obj.spec && obj.spec.rules && obj.spec.rules[0].host })

const servicesGet = async (namespace) => {
  const ret = []
  const service = {}

  const res = await Promise.all([
    client.api.v1.namespaces(namespace).pods.get(),
    // client.api.v1.namespaces(namespace).pods.get({ qs: { labelSelector: 'tier!=job' } }),
    client.api.v1.namespaces(namespace).services.get(),
    client.apis.extensions.v1beta1.namespaces(namespace).ingresses.get(),
  ])

  res[0].body.items.forEach(async (item) => {
    const serviceName = (item.metadata.labels && item.metadata.labels['qcloud-app']) || item.metadata.name
    service[serviceName] = formatPodInfo(item)
  })


  res[1].body.items.forEach(async (item) => {
    console.log(2, item)
    if (service[item.metadata.name]) {
      service[item.metadata.name] = _.assign(service[item.metadata.name], formatServiceInfo(item))
    }
  })

  res[2].body.items.forEach(async (item) => {
    if (service[item.metadata.name]) {
      service[item.metadata.name] = _.assign(service[item.metadata.name], formatIngressInfo(item))
    }
  })

  for (const index in service) {
    if (Object.prototype.hasOwnProperty.call(service, index)) {
      ret.push(service[index])
    }
  }

  return ret
}

module.exports = {
  serviceGet,
  servicesGet,
  podGet,
  serviceCreate,
}
